#!/usr/bin/python
# coding: utf-8
"""
 :copyright: (c) 2011 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""
from glob import glob
import json
import optparse
import os
import os.path
import re
import sys
import site

import werkzeug.serving
import werkzeug.utils


mysql_uri_pattern = re.compile(
               r'^mysql://(?P<user>[^@]+)@(?P<host>[^/]+)/(?P<db>.+)$')


class UsageError(RuntimeError):
    pass


class ConfigModule(object):
    def __init__(self):
        self.config = {}

    def add_config(self, configname, value):
        self.config[configname] = value


def handle_sqlite(data, options, homunculus_config):
    if 'sqlite' not in data['features']:
        return

    homunculus_config.add_config('sqlite', {
        'sqlite_path': os.path.join(options.directory, 'sqlite.db'),
    })


def handle_mysql(data, options, homunculus_config):
    if 'mysql' not in data['features']:
        return

    if not options.mysql_uri:
        raise UsageError(u'the app uses the mysql-feature, '
                         u'please provide --mysql')

    match = mysql_uri_pattern.match(options.mysql_uri)
    if not match:
        raise UsageError(u"%s does not seem to be a valid MySQL-URI"
                                                           % options.mysql_uri)

    homunculus_config.add_config('mysql', {
        'credentials': match.groupdict(),
        'dburl': options.mysql_uri,
    })


def handle_mediadir(data, options, homunculus_config):
    if 'mediadir' not in data['features']:
        return

    mediadir_path = os.path.join(options.directory, 'mediadir')

    if not os.path.isdir(mediadir_path):
        os.makedirs(mediadir_path)

    homunculus_config.add_config('mediadir', {
        'path': mediadir_path,
        'public_path': data['features']['mediadir'].get('public_path'),
    })


def handle_virtualenv(data, options, homunculus_config, recreate):
    if not recreate:
        return

    virtualenv_dir = os.path.join(options.directory, 'virtualenv')

    os.system('virtualenv --no-site-packages %s' % virtualenv_dir)
    for dependency in data['dependencies']:
        os.system('%s/bin/pip install %s' % (virtualenv_dir, dependency))


def run_application(data, options, homunculus_config):
    ve_package_path, = glob(os.path.join(options.directory, 'virtualenv',
                                            'lib', 'python*', 'site-packages'))

    site.addsitedir(ve_package_path)

    def get_sort_key(path):
        return (path.startswith(ve_package_path), path)

    sys.path.sort(key=get_sort_key)
    sys.path.insert(0, '.')

    # currently this is just faked. maybe, in the future, it gets imported
    # for real
    sys.modules['homunculus_config'] = homunculus_config
    sys.modules['homunculus_instancetools'] = homunculus_config

    app = werkzeug.utils.import_string('%s.application' % data['application'])
    werkzeug.serving.run_simple(options.hostname, options.port, app,
                                          use_debugger=True, use_reloader=True)


def main():
    parser = optparse.OptionParser()
    parser.add_option('-p', '--port', dest='port', type="int", default=8080,
                      help="Port for the server to run on.", )
    parser.add_option('-H', '--host', dest='hostname', default="localhost",
                      help="Hostname for the server to run on.")

    parser.add_option('--mysql', dest='mysql_uri',
                      help="When the app uses the 'mysql'-feature, the URI of "
                           "the database, in the form "
                           "'mysql://username@hostname/dbname'")

    parser.add_option('--directory', dest='directory', default=".homunculus",
                      help="Location, where all data necessary for running "
                           "the application is stored.")

    options, args = parser.parse_args()

    try:
        with open('config.json', 'r') as fp:
            data = json.load(fp)
    except Exception, exc:
        parser.error('could not read config.json: %s' % exc)

    if not os.path.exists(options.directory):
        os.makedirs(options.directory)
        recreate = True
    else:
        recreate = os.path.getmtime('config.json') > os.path.getmtime(
                                                             options.directory)

    homunculus_config = ConfigModule()

    homunculus_config.add_config('base', {
        'app_name': data['appname'],
        'revision_name': data['version'],
        'seed': 'this is the seed'.encode("base64").strip(),
        'auth_token': 'this is the auth token'.encode("base64").strip(),
    })

    try:
        handle_sqlite(data, options, homunculus_config)
        handle_mysql(data, options, homunculus_config)
        handle_mediadir(data, options, homunculus_config)
        handle_virtualenv(data, options, homunculus_config, recreate)
    except UsageError, exc:
        parser.error(exc)

    os.system('touch %s' % options.directory)

    run_application(data, options, homunculus_config)


if __name__ == '__main__':
    main()
