#!/usr/bin/python
# coding: utf-8
"""
 :copyright: (c) 2011 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""
from glob import glob
import json
import yaml
import optparse
import os
import os.path
import re
import sys
import site

import werkzeug.serving
from werkzeug.utils import import_string
from werkzeug.wsgi import DispatcherMiddleware, SharedDataMiddleware
from werkzeug.exceptions import NotFound


mysql_uri_pattern = re.compile(
               r'^mysql://(?P<user>[^@]+)@(?P<host>[^/]+)/(?P<db>.+)$')


class Webserver(object):

    def __init__(self):
        self.static_dirs = {}
        self.wsgi_app_names = {}

    def add_static_dir(self, source, mount):
        mount = '/' + mount.lstrip('/')
        self.static_dirs[mount] = source

    def add_wsgi_app(self, modulename, mount):
        mount = '/' + mount.lstrip('/')

        # only name, because the paths may not be initialized yet
        self.wsgi_app_names[mount] = '%s.application' % modulename

    def serve(self, hostname, port):
        # resolve actual wsgi-apps by name
        wsgi_apps = dict((mount, import_string(name))
                                for mount, name in self.wsgi_app_names.items())

        # if an application is mounted on '/', use it. if not, / is 404.
        app = wsgi_apps.get('/', NotFound())

        app = SharedDataMiddleware(app, self.static_dirs)
        app = DispatcherMiddleware(app, wsgi_apps)

        from pprint import pprint
        pprint(wsgi_apps)

        werkzeug.serving.run_simple(hostname, port, app,
                                          use_debugger=True, use_reloader=True)


class UsageError(RuntimeError):
    pass


class ConfigModule(object):
    def __init__(self):
        self.config = {}

    def add_config(self, configname, value):
        self.config[configname] = value


def handle_sqlite(server, data, options, homunculus_config):
    if 'sqlite' not in data['features']:
        return

    homunculus_config.add_config('sqlite', {
        'sqlite_path': os.path.join(options.directory, 'sqlite.db'),
    })


def handle_mysql(server, data, options, homunculus_config):
    if 'mysql' not in data['features']:
        return

    if not options.mysql_uri:
        raise UsageError(u'the app uses the mysql-feature, '
                         u'please provide --mysql')

    match = mysql_uri_pattern.match(options.mysql_uri)
    if not match:
        raise UsageError(u"%s does not seem to be a valid MySQL-URI"
                                                           % options.mysql_uri)

    homunculus_config.add_config('mysql', {
        'credentials': match.groupdict(),
        'dburl': options.mysql_uri,
    })


def handle_mediadir(server, data, options, homunculus_config):
    if 'mediadir' not in data['features']:
        return

    mediadir_path = os.path.join(options.directory, 'mediadir')
    public_path = data['features']['mediadir'].get('public_path')

    if not os.path.isdir(mediadir_path):
        os.makedirs(mediadir_path)

    if public_path:
        server.add_static_dir(mediadir_path, public_path)

    homunculus_config.add_config('mediadir', {
        'path': mediadir_path,
        'public_path': public_path,
    })


def handle_static(server, data, options, homunculus_config):
    for source, mount in data['features'].get('static', {}).items():
        server.add_static_dir(source, mount)


def handle_python(server, data, options, homunculus_config, recreate):
    if 'python' not in data['features']:
        return

    if recreate:
        virtualenv_dir = os.path.join(options.directory, 'virtualenv')

        # create virtual environment
        os.system('virtualenv --no-site-packages %s' % virtualenv_dir)

        # install dependencies
        for dependency in data['features']['python'].get('dependencies'):
            os.system('%s/bin/pip install %s' % (virtualenv_dir, dependency))

    # mount wsgi apps
    wsgi_modules = data['features']['python'].get('wsgi', {}).items()
    for modulename, mount in wsgi_modules:
        server.add_wsgi_app(modulename, mount)


def run_application(server, data, options, homunculus_config):
    ve_package_path, = glob(os.path.join(options.directory, 'virtualenv',
                                            'lib', 'python*', 'site-packages'))

    site.addsitedir(ve_package_path)

    def get_sort_key(path):
        return (path.startswith(ve_package_path), path)

    sys.path.sort(key=get_sort_key)
    sys.path.insert(0, '.')

    # currently this is just faked. maybe, in the future, it gets imported
    # for real
    sys.modules['homunculus_config'] = homunculus_config
    sys.modules['homunculus_instancetools'] = homunculus_config

    server.serve(options.hostname, options.port)


def main():
    parser = optparse.OptionParser()
    parser.add_option('-p', '--port', dest='port', type="int", default=8080,
                      help="Port for the server to run on.", )
    parser.add_option('-H', '--host', dest='hostname', default="localhost",
                      help="Hostname for the server to run on.")

    parser.add_option('--mysql', dest='mysql_uri',
                      help="When the app uses the 'mysql'-feature, the URI of "
                           "the database, in the form "
                           "'mysql://username@hostname/dbname'")

    parser.add_option('--directory', dest='directory', default=".homunculus",
                      help="Location, where all data necessary for running "
                           "the application is stored.")

    options, args = parser.parse_args()

    if os.path.exists('homunculus.json'):
        with open('homunculus.json', 'r') as fp:
            data = json.load(fp)
            mtime = os.path.getmtime('homunculus.json')
    elif os.path.exists('homunculus.yaml'):
        with open('homunculus.yaml', 'r') as fp:
            data = yaml.load(fp)
            mtime = os.path.getmtime('homunculus.yaml')
    else:
        parser.error('need homunculus.[json|yaml]')

    if not os.path.exists(options.directory):
        os.makedirs(options.directory)
        recreate = True
    else:
        recreate = mtime > os.path.getmtime(options.directory)

    homunculus_config = ConfigModule()

    homunculus_config.add_config('base', {
        'app_name': 'Appname',
        'revision_name': data['version'],
        'seed': 'this is the seed'.encode("base64").strip(),
        'auth_token': 'this is the auth token'.encode("base64").strip(),
    })

    server = Webserver()

    try:
        handle_sqlite(server, data, options, homunculus_config)
        handle_mysql(server, data, options, homunculus_config)
        handle_mediadir(server, data, options, homunculus_config)
        handle_python(server, data, options, homunculus_config, recreate)
    except UsageError, exc:
        parser.error(exc)

    os.system('touch %s' % options.directory)

    run_application(server, data, options, homunculus_config)


if __name__ == '__main__':
    main()
