# coding: utf-8
"""

Media Directory
===============

With ::

    "features": {
        "mediadir": {
            'public_path': '/some/public/path'
        }
        #...
    }


an application can request a directory, writable by the application, that will
be served by the webserver.

The directory will stay between revisions.

The following configuration will be provided: ::

    {
        "path": "/local/path/of/the/mediadir",
        "public_path": "/some/public_path/path"
    }


:copyright: (c) 2011 Philipp Benjamin Köppchen
:license: GPLv3, see LICENSE for more details.
"""
from __future__ import absolute_import

import shutil
from . import feature_added, feature_backuped, alias_added


@feature_added.connect_via(intern('mediadir'))
def on_revision_added_provide_mediadir(name, hom_env, app_env, revision_env,
                                                               feature_config):
    """ Creates a Mediadir and the config

    revision_env
        the RevisionEnvironment
    """
    path = app_env.get_feature_path('mediadir')
    path.makedirs()

    alias = feature_config.get('public_path', 'media')

    revision_env.logger('mounting mediadir to %s' % alias)
    alias_added.send(hom_env, app_env=app_env,
                              revision_env=revision_env,
                              alias=alias,
                              path=path)

    revision_env.add_config('mediadir', {
        'path': path,
        'public_path': feature_config.get('public_path'),
    })


@feature_backuped.connect_via(intern('mediadir'))
def on_revision_backuped_save_mediadir(name, hom_env, app_env, revision_env,
                                                   backup_env, feature_config):

    source_path = app_env.get_feature_path('mediadir')
    target_path = backup_env.get_feature_path('mediadir')

    #target_path.makedirs()
    shutil.copytree(source_path, target_path)
