# coding: utf-8
"""
 :copyright: (c) 2011 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""
from __future__ import absolute_import

from os import urandom
from . import revision_added, revision_removed, alias_added, feature_added


@revision_added.connect
def on_revision_added_extract_package(hom_env, app_env, revision_env, package,
                                                                     **kwargs):
    """Extracts the package into $path/raw"""
    revision_env.revision_path.makedirs()
    package.extract(revision_env.revision_path.join('raw'))


@revision_added.connect
def on_revision_added_provide_base_config(hom_env, app_env, revision_env,
                                                                     **kwargs):
    """ Makes app_name, revision_name and seeds available for config
    """
    revision_env.add_config('base', {
        'app_name': app_env.app_name,
        'revision_name': revision_env.revision_name,
        'seed': urandom(32).encode("base64").strip(),
        'auth_token': urandom(32).encode("base64").strip(),
    })


@revision_added.connect
def on_revision_added_handle_features(hom_env, app_env, revision_env, package,
                                                                     **kwargs):
    """ Handles the feature signals
    """
    for name, config in package.features.items():
        name = intern(name.encode('utf-8'))
        feature_added.send(name, hom_env=hom_env,
                                 app_env=app_env,
                                 revision_env=revision_env,
                                 feature_config=config)


@feature_added.connect_via(intern('static'))
def mount_static_files(feature_name, hom_env, app_env, revision_env,
                                                     feature_config, **kwargs):
    """Dispatch creation of static files to webserver handlers
    """
    for rel_path, alias in feature_config.items():
        # fixing leading slashes
        rel_path = rel_path.lstrip('/')
        alias = '/' + alias.lstrip('/')

        revision_env.logger('mounting <package>/%s to %s' % (rel_path, alias))

        path = revision_env.revision_path.join('raw', rel_path)
        alias_added.send(hom_env, app_env=app_env,
                                  revision_env=revision_env,
                                  alias=alias,
                                  path=path)


# revision removal handlers


@revision_removed.connect
def handle_revision_removal(hom_env, app_env, revision_env, **kwargs):
    # be tolerant, to allow to remove revisions whose files have been removed
    # manually
    if revision_env.revision_path.exists:
        revision_env.revision_path.remove()
