# coding: utf-8
"""

MySQL Database
==============

With ::

  "features": {
      "mysql": true
      # ...
  }

a package can request a MySQL Database. The database will be freshly created if
none exists for this application, and stay between revisions.

The following configuration will be provided: ::

  {
      "credentials": {
          "host": "localhost",
          "db": "hom_conceptor",
          "user": "hom_conceptor"
      },
      "dburl": "mysql://hom_conceptor@localhost/hom_conceptor"
  }


:copyright: © 2011 Philipp Benjamin Köppchen
:license: GPLv3, see LICENSE for more details.

"""
from __future__ import absolute_import

import subprocess
from . import feature_added, feature_backuped


class Database(object):
    def __init__(self, config):
        # importing locally, to not enforce existence of library
        from MySQLdb import connect

        self.conn = connect(**config)
        self.cursor = self.conn.cursor()
        self.config = config

    def execute(self, query, *args):
        self.cursor.execute(query, args)
        rows = self.cursor.fetchall()
        self.conn.commit()
        return rows

    def close(self):
        self.conn.close()

    def dump(self, dbname, filename):
        args = ['mysqldump', '-u', self.config['user'],
                             '-h', self.config['host']]
        if 'passwd' in self.config:
            args += ['-p', self.config['passwd']]

        args += [dbname]

        with open(filename, 'wb') as fp:
            subprocess.check_call(args, stdout=fp)


def _get_db_name(app_name):
    # values for revision database
    dbname = 'hom_%s' % (app_name.replace("`", "").replace("'", ""))

    # TODO: this is a little too hackish. no database names longer than 16
    # chars allowed
    return dbname[:16]


@feature_added.connect_via(intern('mysql'))
def on_revision_added_provide_mysql(name, hom_env, app_env, revision_env,
                                                               feature_config):
    revision_env.logger('providing mysql...')

    # default values
    handler_config = {'host': 'localhost',
                      'user': 'homunculus'}

    handler_config.update(
                     hom_env.get_handler_config('handlers.mysql.handle_mysql'))

    database = Database(handler_config)

    dbname = _get_db_name(revision_env.app_name)
    username = dbname
    hostname = 'localhost'

    if dbname in [d for d, in database.execute("show databases")]:
        revision_env.logger("database %s already exists, using it." % dbname)
    else:
        revision_env.logger("creating databse %s..." % dbname)
        # via string formatting, because dbnames are not standard arguments
        database.execute("create database `%s`" % dbname)
        database.execute("grant all privileges on `%s`.* to '%s'@'localhost'"
                                                          % (dbname, username))

    revision_env.add_config('mysql', {
        'credentials': {
            'db': dbname,
            'user': username,
            'host': hostname,
        },
        'dburl': 'mysql://%s@%s/%s' % (username, hostname, dbname),
    })

    database.close()


@feature_backuped.connect_via(intern('mysql'))
def on_feature_backuped_dump_mysql(name, hom_env, app_env, revision_env,
                                                   backup_env, feature_config):

    handler_config = hom_env.get_handler_config('handlers.mysql.handle_mysql')
    database = Database(handler_config)
    dbname = _get_db_name(revision_env.app_name)

    dump_dirname = backup_env.get_feature_path('mysql')
    dump_dirname.makedirs()

    database.dump(dbname, dump_dirname.join('dump.sql'))
