# coding: utf-8
"""
 :copyright: (c) 2011 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""
from __future__ import with_statement

import glob
import os
import os.path
import shutil
import subprocess
import tempfile
import zipfile


class ShellCommandError(Exception):
    def __init__(self, msg, out, err):
        Exception.__init__(self, msg)
        self.out = out
        self.err = err


class ShellCommand(object):
    """ Encapsulates a shell command

    example:
    >>> cmd = ShellCommand('rm', '-rf', '%(0)')
    >>> cmd('/tmp/xxx')  # results in `rm -rf /tmp/xxx`
    """
    def __init__(self, *args):
        """ Constructor

        *args
            the subprocess's arguments, just like for Popen. It is possible
            to use placeholders like %(0)s, these will be replaced with the
            arguments given to __call__, when called
        """
        self.args = args

    def __call__(self, *values):
        """ Executes the shell command.

        *values
            these are used to replace the placeholdes given in the constructor.

        raises
            `ShellCommandError`, when the command results in a non-zero
            exit code.
        """
        values = dict((str(index), value)
                              for index, value in enumerate(values))
        processargs = [arg % values for arg in self.args]
        proc = subprocess.Popen(processargs, stdout=subprocess.PIPE,
                                             stderr=subprocess.PIPE)
        out, err = proc.communicate()

        if proc.returncode != 0:
            msg = '%r returned non-zero exit status %s.'
            raise ShellCommandError(msg % (processargs, proc.returncode),
                                                                      out, err)


def makedirs(*pathfragments):
    path = os.path.join(*pathfragments)
    if not os.path.isdir(path):
        os.makedirs(path)


def zip_path(path, filename):
    rootlen = len(path) + 1

    zf = zipfile.ZipFile(filename, 'w', zipfile.ZIP_DEFLATED)
    try:
        for base, _, filenames in os.walk(path):
            for filename in filenames:
                filename = os.path.join(base, filename)
                zf.write(filename, filename[rootlen:])
    finally:
        zf.close()


class Virtualenv(object):
    virtualenv = ShellCommand('virtualenv', '--no-site-packages', '%(0)s')

    # call via python interpreter, to avoid problems with shebang with spaces,
    # that may arise, when the virtualenv is located in a directory with spaces
    # in the name
    pip = ShellCommand('%(0)s/bin/python', '%(0)s/bin/pip', 'install', '%(1)s')

    def __init__(self, target):
        self.target = target

    def create(self):
        return self.virtualenv(self.target)

    def install(self, requirement):
        return self.pip(self.target, requirement)


class Tempdir(object):

    def __enter__(self):
        self.path = Path(tempfile.mkdtemp())
        return self.path

    def __exit__(self, type, value, traceback):
        self.path.remove()


def path_property(func):
    return property(lambda *args, **kwargs: Path(func(*args, **kwargs)))


class Path(unicode):

    def join(self, *args):
        return Path(os.path.join(self, *args))

    def open(self, *args):
        return open(self, *args)

    def get_content(self):
        with self.open('r') as fp:
            return fp.read()

    def set_content(self, data):
        with self.open('w') as fp:
            fp.write(data)

    content = property(get_content, set_content)

    def append(self, data):
        with self.open('a') as fp:
            fp.write(data)

    def makedirs(self):
        if not self.isdir:
            os.makedirs(self)

    def remove(self):
        if self.isdir:
            shutil.rmtree(self)
        else:
            os.remove(self)

    def globjoin(self, pattern):
        """ extends the current path by trying to resolve the glob pattern.

        raises ValueError, if the pattern cannot be resolved unambigously or
        at all.

        >>> Path('/etc/passw?')
        <<< Path('/etc/passwd')
        """
        matching = glob.glob(self.join(pattern))
        if not matching:
            raise ValueError('cannot resolve path %r' % self)
        elif len(matching) > 1:
            raise ValueError('path %r is ambigous' % self)
        else:
            return Path(matching[0])

    def glob(self, pattern):
        """ returns a list of all files matching the pattern.

        >>> Path('/etc').glob('passw?')
        <<< ['/etc/passwd']
        """
        return [Path(item) for item in glob.glob(self.join(pattern))]

    abspath = path_property(os.path.abspath)
    basename = path_property(os.path.basename)
    dirname = path_property(os.path.dirname)
    expanduser = path_property(os.path.expanduser)
    expandvars = path_property(os.path.expandvars)
    realpath = path_property(os.path.expandvars)

    exists = property(os.path.exists)
    isfile = property(os.path.isfile)
    isdir = property(os.path.isdir)
    isabs = property(os.path.isabs)
    islink = property(os.path.islink)
    ismount = property(os.path.ismount)
