# coding: utf-8
"""

SQLite Database
===============

With ::

  "features: {
      "sqlite": true
      # ...
  }

a package can request a SQLite Database. The database will stay between
revisions.

The following configuration will be provided: ::

  {
     "sqlite_path": "/path/to/sqlite/file.db"
  }

:copyright: (c) 2011 Philipp Benjamin Köppchen
:license: GPLv3, see LICENSE for more details.

"""
from __future__ import absolute_import
import shutil
from . import feature_added, feature_backuped


@feature_added.connect_via(intern('sqlite'))
def on_revision_added_provide_sqlite(name, hom_env, app_env, revision_env,
                                                               feature_config):
    """ Provides the configuration and database files for the sqlite-feature

    revision_env
        the RevisionEnvironment
    """
    revision_env.logger('providing sqlite...')

    path = app_env.get_feature_path('sqlite')
    path.makedirs()

    revision_env.add_config('sqlite', {
        'sqlite_path': path.join('sqlite.db'),
        'dburl': 'sqlite:///%s' % path.join('sqlite.db'),
    })


@feature_backuped.connect_via(intern('sqlite'))
def on_feature_backuped_dump_sqlite(name, hom_env, app_env, revision_env,
                                                   backup_env, feature_config):

    source_file = app_env.get_feature_path('sqlite').join('sqlite.db')

    # file may not exist, because the app was never started. fine anyway.
    if not source_file.exists:
        return

    target_path = backup_env.get_feature_path('sqlite')
    target_path.makedirs()

    shutil.copy(source_file, target_path)
