# coding: utf-8
"""
 :copyright: (c) 2011 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""
from __future__ import absolute_import, with_statement

from datetime import datetime
import os
import os.path
import re
from StringIO import StringIO
from time import sleep
import threading
import types
import yaml
import zipfile

from flask import request, json, current_app
from flask.ext.sqlalchemy import SQLAlchemy
from sqlalchemy.orm import validates, deferred
from sqlalchemy.orm.exc import NoResultFound
from sqlalchemy.schema import UniqueConstraint

#from . import app as app
from . import handlers
from .helpers import ShellCommandError, Tempdir, zip_path


DOMAIN_PATTERN = re.compile(r'^([a-z0-9-]+\.)+[a-z0-9-]+$', re.IGNORECASE)
NAME_PATTERN = re.compile(r'^[a-z0-9 _.-]+$', re.IGNORECASE)


db = SQLAlchemy()


def commit():
    db.session.commit()


def rollback():
    db.session.rollback()


def asynchronous(func):
    def wrapper(*args, **kwargs):
        environ = request.environ
        cfg = current_app.config

        def task():
            from homunculus_server import create_app
            app = create_app(cfg)
            with app.request_context(environ):
                return func(*args, **kwargs)

        t = threading.Thread(target=task)
        t.start()

    return wrapper


def create_homunculus_environment():
    return handlers.HomunculusEnvironment(current_app.config)


class ValidationError(Exception):
    def __init__(self, field, error):
        self.field = field
        self.error = error
        Exception.__init__(self, u"%s: %s" % (field, error))

    def to_dict(self):
        return {self.field: self.error}


class App(db.Model):
    __tablename__ = 'App'

    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(50), unique=True, nullable=False)

    domain = db.Column(db.String(150), nullable=False, default='')

    current_revision_id = db.Column(db.Integer, db.ForeignKey('Revision.id'),
                                                                 nullable=True)

    current_revision = db.relationship('Revision',
                        primaryjoin='App.current_revision_id == Revision.id')

    # revisions per backref

    @classmethod
    def get_by_name(cls, name):
        return cls.query.filter_by(name=name).one()

    @classmethod
    def find_all(cls):
        return cls.query.all()

    def __init__(self, name):
        self.name = name
        db.session.add(self)

    @validates('name')
    def validate_name(self, key, value):
        if not value or not value.strip():
            raise ValidationError(u'name', u'must be given')

        if len(value) < 3:
            raise ValidationError(u'name',
                                  u'must be at least 3 characters long')

        if not NAME_PATTERN.match(value):
            raise ValidationError(u'name',
                                  u'contains invalid characters')

        value = value.strip()

        otherapp = App.query.filter_by(name=value).first()
        if otherapp and otherapp != self:
            raise ValidationError(u'name', u'name already taken')

        return value

    @validates('domain')
    def validate_domain(self, key, value):
        domains = [d for d in value.split(' ') if d.strip()]

        for domain in domains:
            if not DOMAIN_PATTERN.match(domain):
                raise ValidationError(u'domains',
                    u'"{0}" does not seem to be a valid domain'.format(domain))

        return value

    def set_current_revision(self, revision_name):
        revision = Revision.get_by_appname_and_revisionname(self.name,
                                                             revision_name)
        if not revision.can_be_current:
            raise Exception("revision cannot be set as current")

        handlers.current_revision_set.send(create_homunculus_environment(),
                                            app_env=self._environment,
                                            revision_env=revision._environment)
        self.current_revision = revision


    def create_backup(self):
        if not self.current_revision:
            raise Exception("no current revision")

        with Tempdir() as backup_path:
            backup_environment = handlers.BackupEnvironment(backup_path)
            backup_environment.path.makedirs()

            package = Package(StringIO(self.current_revision.data))



            for name, config in package.features.items():
                name = intern(name.encode('utf-8'))
                handlers.feature_backuped.send(name,
                                             hom_env=create_homunculus_environment(),
                                             app_env=self._environment,
                                             revision_env=self.current_revision._environment,
                                             backup_env=backup_environment,
                                             feature_config=config)

            zip_path(backup_path, '/tmp/backup.zip')

        with open('/tmp/backup.zip') as fp:
            return Backup(self, fp.read())

    def add_revision(self, data):
        package = Package(StringIO(data))

        try:
            Revision.get_by_appname_and_revisionname(self.name,
                                                               package.version)
        except NoResultFound:
            pass
        else:
            raise PackageError("there is already a revision %s"
                                                             % package.version)
        return Revision(self, package.version, data)

    def to_dict(self):
        revision = self.current_revision
        return {
            'name': self.name,
            'domain': self.domain,
            'status': revision and revision.status,
            'current_revision_name': revision and revision.name,
            'generic_domain': self.generic_domain,
        }

    @property
    def generic_domain(self):
        clean_name = re.sub('[^a-z0-9-]', '', self.name.lower())
        return '%s.%s' % (clean_name,
                         current_app.config.get('generic_domain', 'localhost'))

    @property
    def _environment(self):
        domains = [d for d in self.domain.split(' ') if d.strip()]
        domains.append(self.generic_domain)

        return handlers.ApplicationEnvironment(current_app.config['BASEDIR'],
                                   self.name, domains)

    def delete(self):
        self.current_revision = None
        # ...or else this version could not be deleted

        for revision in self.revisions:
            revision.delete()

        handlers.app_removed.send(create_homunculus_environment(),
                                  app_env=self._environment)
        db.session.delete(self)


class Revision(db.Model):
    __tablename__ = 'Revision'

    __table_args__ = (
        UniqueConstraint('app_id', 'name'),
    )

    id = db.Column(db.Integer, primary_key=True)

    name = db.Column(db.String(50), nullable=False)
    status = db.Column(db.String(50), default=u'preparing', nullable=False)

    app_id = db.Column(db.Integer, db.ForeignKey('App.id'), nullable=False)
    app = db.relationship('App', primaryjoin='Revision.app_id == App.id',
                               backref=db.backref('revisions', lazy='dynamic'))

    data = deferred(db.Column(db.LargeBinary, nullable=False))
    log = deferred(db.Column(db.UnicodeText, nullable=False, default=u''))

    @classmethod
    def get_by_id(cls, id_):
        return cls.query.get(id_)

    @classmethod
    def get_by_appname_and_revisionname(cls, app_name, revision_name):
        return cls.query.filter(Revision.name == revision_name,
                                Revision.app_id == App.id,
                                App.name == app_name).one()

    def __init__(self, app, name, data):
        self.app = app
        self.name = name
        self.data = data
        db.session.add(self)

    @validates('name')
    def validate_name(self, key, value):
        if not value.strip():
            raise ValidationError(u'name', u'must be given')
        if not NAME_PATTERN.match(value):
            raise ValidationError(u'name', u'contains invalid characters')
        # Why is the duplicate check not here?
        return value.strip()

    @property
    def _environment(self):
        return handlers.RevisionEnvironment(current_app.config['BASEDIR'],
                                            self.app.name,
                                            self.name,
                                            self.logger)

    @property
    def is_current(self):
        return self.app.current_revision == self

    @property
    def can_be_current(self):
        return self.status == 'ready'

    @property
    def is_deletable(self):
        return not self.is_current

    @property
    def config(self):
        return self._environment.get_config()

    @property
    def server_log(self):
        return self._environment.get_server_log()

    def to_dict(self):
        return {
            'app': self.app.name,
            'name': self.name,
            'status': self.status,
            'log': self.log,
            'server_log': self.server_log,
        }

    def delete(self):
        handlers.revision_removed.send(create_homunculus_environment(),
                                            app_env=self.app._environment,
                                            revision_env=self._environment)
        db.session.delete(self)

    def prepare(self):
        @asynchronous
        def prepare_revision(id_):
            sleep(1)
            revision = Revision.get_by_id(id_)
            revision._prepare()
        prepare_revision(self.id)

    def _prepare(self):
        revision_env = self._environment
        revision_env.revision_path.makedirs()
        package = Package(StringIO(self.data))

        try:
            handlers.revision_added.send(create_homunculus_environment(),
                                            app_env=self.app._environment,
                                            revision_env=revision_env,
                                            package=package)
        except ShellCommandError, exc:
            self.log += repr(exc) + '\n'
            self.log += 'stdout:' + '\n' + exc.out + '\n'
            self.log += 'stderr:' + '\n' + exc.err + '\n'
            self.status = u'error'
            commit()
            raise
        except Exception, exc:
            self.log += u'Exception: %r' % exc
            self.status = u'error'
            commit()
            raise
        else:
            self.status = u'ready'
        commit()

    def logger(self, line):
        self.log += unicode(line + '\n')
        commit()


class Backup(db.Model):
    __tablename__ = 'Backup'

    id = db.Column(db.Integer, primary_key=True)

    app_id = db.Column(db.Integer, db.ForeignKey('App.id'), nullable=False)
    app = db.relationship('App', primaryjoin='Backup.app_id == App.id',
                               backref=db.backref('backups', lazy='dynamic'))

    datetime = db.Column(db.DateTime, nullable=False, default=datetime.now)
    data = deferred(db.Column(db.LargeBinary, nullable=False))

    @classmethod
    def get_by_id(cls, id_):
        return cls.query.get(id_)

    def __init__(self, app, data):
        self.app = app
        self.data = data
        db.session.add(self)


class PackageError(RuntimeError):
    """Exception, to notify about a problem with Parsing a Package"""
    pass


class Package(object):
    """ A Packaged Revision.

    has some interesting properties:

    version
        version of the application, e.g. "0.0.23"
    features
        dictionary of features, in the form <featurename>: <config>.
    """

    def __init__(self, data):
        """Constructor.

        data
            the raw data of the package, as a filelike (e.g a StringIO)
        raises
            PackageError, when a problem with parsing the data or interpreting
            the config.json is encountered
        """
        try:
            self.zip = zipfile.ZipFile(data)
            # decode, to make simplejson answer unicode
            config = self._load_config()
        except Exception, exc:
            raise PackageError('problem while parsing the package: %s' % exc)

        def set_config(key, default, allowed_types):
            """Helper, to adopt a configuration value"""
            value = config.get(key, default)
            if not isinstance(value, allowed_types):
                raise PackageError('Problem with configuration value %r' % key)
            setattr(self, key, value)

        set_config('version', None, types.StringTypes)
        set_config('features', {}, (dict,))

    def _zipped_file(self, filename):
        try:
            return self.zip.read(filename)
        except KeyError:
            return None

    def _load_config(self):
        content = self._zipped_file('homunculus.json')
        if content:
            return json.loads(content.decode('utf-8'))

        content = self._zipped_file('homunculus.yaml')
        if content:
            return yaml.load(content)

        raise PackageError('No config file was found')

    def extract(self, directory):
        """Extracts the Zipfile into a directory"""
        os.mkdir(directory)

        for filename in sorted(self.zip.namelist()):
            targetname = os.path.join(directory, filename)
            if filename.endswith("/"):
                os.mkdir(targetname)
            else:
                with open(targetname, 'wb') as fp:
                    fp.write(self.zip.read(filename))
