# coding: utf-8
"""
 :copyright: (c) 2011 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""
from __future__ import with_statement, absolute_import

import random

from werkzeug.exceptions import BadRequest
from flask import Blueprint, render_template, request, session, jsonify, \
                                                                  make_response
from . import model
from .model import App, Backup, Revision, ValidationError

#from . import app as bp

bp = Blueprint('controller', __name__)


@bp.route('/')
def app_list():
    """ Shows a list of all applications
    """
    return render_template('app_list.html.jinja', apps=App.find_all())


@bp.route('/apps/<string:app_name>/')
def app_details(app_name):
    """ Shows the application's details
    """
    return render_template('app_details.html.jinja',
                           app=App.get_by_name(app_name))


@bp.route('/apps/<string:app_name>/<string:revision_name>/')
def revision_details(app_name, revision_name):
    """ Shows the revision's details
    """
    app = App.get_by_name(app_name)
    revision = Revision.get_by_appname_and_revisionname(app_name,
                                                        revision_name)

    return render_template('revision_details.html.jinja', app=app,
                                                          revision=revision)


@bp.route('/api/')
def api_entry_point():
    """ The entry point for the api. Maybe in the future we get a little resty
    """
    return jsonify({
        'xsrf_token': session['xsrf_token'],
    })


@bp.route('/api/apps/')
def index_apps():
    """ an Overview for all installed Applications (json) """
    return jsonify({
        'applications': [app.to_dict() for app in App.find_all()],
    })


@bp.route('/api/apps/<string:app_name>/')
def get_app(app_name):
    """ Details of a specific application (json) """
    app = App.get_by_name(app_name)

    result = app.to_dict()
    result.update({
        'revisions': [r.name for r in app.revisions],
    })
    return jsonify(result)


@bp.route('/api/apps/<string:app_name>/revisions/<string:revision_name>/')
def get_revision(app_name, revision_name):
    """ Details of a specific revision (json) """
    revision = Revision.get_by_appname_and_revisionname(app_name,
                                                        revision_name)
    return jsonify(revision.to_dict())


@bp.route('/api/apps/new/', methods=["POST"])
def new_app():
    """ Creates a new application

    expects the following form values:

    name
        name of the application

    returns a json object, with the following keys

    success
        true, if the creation succeeded, false otherwise
    errors
        an object formfield -> errortext
    """
    try:
        app = App(request.form['name'])
        return jsonify({'success': True, 'errors': {}})
    except ValidationError, exc:
        return jsonify({'success': False, 'errors': exc.to_dict()})


@bp.route('/apps/<string:app_name>/edit/', methods=["POST"])
def edit_app(app_name):
    """ Updates an application

    excepts the following form values:

    revision_name (optional)
        name of the revision, that should be set as the current revision
    domains (optional)
        a space separated list of domains under which the application should be
        avialable

    returns a json object, with the following keys

    success
        true, if the update succeeded, false otherwise
    errors
        an object formfield -> errortext
    """
    app = App.get_by_name(app_name)

    if 'domains' in request.form:
        try:
            app.domain = request.form['domains']
        except ValidationError, exc:
            return jsonify({'success': False, 'errors': exc.to_dict()})

        # regenerate the configuration, if necessary
        if app.current_revision:
            app.set_current_revision(app.current_revision.name)

    return jsonify({'success': True, 'errors': {}})


@bp.route('/apps/<string:app_name>/', methods=["DELETE"])
def delete_app(app_name):
    """ Deletes an application

    returns a json object with the following keys

    success
        true, if deletion succeeded, false otherwise
    """
    app = App.get_by_name(app_name)
    app.delete()
    return jsonify({'success': True})


@bp.route('/apps/<string:app_name>/backups/', methods=["POST"])
def new_backup(app_name):
    """ Creates a backup of the application's current state

    returns a json object with the following keys

    success
        true, if creation succeeded, false otherwise
    """
    app = App.get_by_name(app_name)
    app.create_backup()
    return jsonify({'success': True})


@bp.route('/apps/<string:app_name>/backups/<int:backup_id>/')
def get_backup(app_name, backup_id):
    """ returns a backup for download
    """
    app = App.get_by_name(app_name)
    backup = Backup.get_by_id(backup_id)

    filename = '%s_%s.zip' % (app_name,
                              backup.datetime.strftime('%Y-%m-%m-%H%M'))

    response = make_response(backup.data)
    response.headers.add('Content-Disposition',
                         'attachment',
                         filename=filename)
    return response


@bp.route('/apps/<string:app_name>/revisions/new/', methods=["POST"])
def new_revision(app_name):
    """ Creates a new revision

    excepts the following form values:

    package:
        package file of the revision

    returns a json object with the following keys

    success
        true, if creation succeeded, false otherwise
    """
    app = App.get_by_name(app_name)
    packagedata = request.files['package'].read()

    def jsonp(data):
        from flask import json
        return """<!doctype html>
                  <html>
                    <head>
                      <title>IFrame javascript response</title>
                      <script type="text/javascript">
                          var json_content = %s;
                      </script>
                    </head>
                    <body>
                      <pre>%s</pre>
                    </body>
                  </html>""" % (json.dumps(data), json.dumps(data))

    try:
        revision = app.add_revision(packagedata)
        model.commit()
    except model.PackageError, exc:
        model.rollback()
        return jsonp({'success': False, 'error': unicode(exc)})

    revision.prepare()
    return jsonp({'success': True})


@bp.route('/apps/<string:app_name>/revisions/<string:revision_name>/current/',
                                                              methods=["POST"])
def set_revision_current(app_name, revision_name):
    """ Sets a revision as the current revision
    """
    app = App.get_by_name(app_name)
    app.set_current_revision(revision_name)
    return jsonify({'success': True})


@bp.route('/apps/<string:app_name>/revisions/<string:revision_name>/',
                                                            methods=["DELETE"])
def delete_revision(app_name, revision_name):
    """ Deletes a revision

    returns a json object with the following keys

    success
        true, if creation succeeded, false otherwise
    """
    revision = Revision.get_by_appname_and_revisionname(app_name,
                                                         revision_name)
    revision.delete()
    return jsonify({'success': True})


@bp.route('/apps/<string:app_name>/revisions/<string:revision_name>/package/')
def download_revision(app_name, revision_name):
    """ Returns the package that was used for creating a revision as download
    """
    revision = Revision.get_by_appname_and_revisionname(app_name,
                                                        revision_name)
    response = make_response(revision.data)
    response.headers.add('Content-Disposition', 'attachment',
                         filename='%s_%s.zip' % (app_name, revision_name))
    return response


@bp.context_processor
def add_xsrf_token():
    return {'xsrf_token': session['xsrf_token']}


@bp.before_request
def protect_against_xsrf():
    if not 'xsrf_token' in session:
        session['xsrf_token'] = str(random.randint(0, 10 ** 10))

    if (request.method == 'POST'
                  and not request.path.startswith('/migrations')
                  and session['xsrf_token'] != request.form.get('xsrf_token')):

        raise BadRequest("Request seems to be forged.")


@bp.after_request
def after_request(response):
    """Commits the database again at the end of the request."""
    model.commit()
    return response
