# coding: utf-8
"""
Directory Layout

/webserver/<servername>/*   # virtualhost configurations
/apps/<app>                 # 'per application'-directory
    /<feature>              # directories for features that stay between
                            # revisions, such as database files, media
                            # directories
    /revisions/<revision>   # 'per revision'-directory
        /runner.wsgi        # wsgi runner file, for usage with webservers
        /config/*.json      # configurations, added via add_config
        /raw                # extracted package
        /virt/              # virtualenv for dependencies and instancetools



Signal Sequence:

    revision_added
        wsgi_application_added
        alias_added
        feature_added



 :copyright: (c) 2011 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""
from __future__ import absolute_import, with_statement

import os.path
import blinker
from flask import json

from ..helpers import Path, makedirs


signal_namespace = blinker.Namespace()


revision_added = signal_namespace.signal('revision-added', doc="""
Signal for handling the addition of a revision.

Sender is the `HomunculusEnvironment`.

Arguments are:

app_env
    `ApplicationEnvironment` for the added revision.

revision_env
    `RevisionEnvironment` for the added revision.

package
    The uploaded Package, from which the new Revision should be set up.
""")


revision_removed = signal_namespace.signal('revision-removed', doc="""
Signal for handling the removal of a revision.

Sender is the `HomunculusEnvironment`.

Arguments are:

app_env
    `ApplicationEnvironment` for the removed revision.

revision_env
    `RevisionEnvironment` for the removed revision.
""")


app_removed = signal_namespace.signal('app-removed')
current_revision_set = signal_namespace.signal('current-revision-set')
webserver_reload = signal_namespace.signal('webserver-reload')

feature_added = signal_namespace.signal('feature-added')
feature_backuped = signal_namespace.signal('feature-backuped')

wsgi_application_added = signal_namespace.signal('wsgi-application-added')
alias_added = signal_namespace.signal('alias-added')


class HomunculusEnvironment(object):

    def __init__(self, homunculus_config):
        self.config = homunculus_config

    def get_handler_config(self, handler_name):
        """ Returns the configuration for a handler.

        handler_name:
            name of the handler, e.g: 'handlers.mysql.handle_mysql'
        returns
            a dictionary containing the configuration values for the handler.
            The dictionary may be empty.
        """
        return self.config['HANDLER_CONFIG'].get(handler_name, {})


class ApplicationEnvironment(object):
    """ Environment of an Application """

    def __init__(self, base_path, app_name, domains):
        """ Constructor

        base_path
            homunculus's root path
        app_name
            name of the application
        domains
            domains, under which the application should be served
        """
        self.base_path = Path(base_path)
        self.app_name = app_name
        self.domains = domains

    def get_virtualhost_path(self, servername):
        """ Returns the path, under which the virtualhost configuration for
        this app on a specific webserver may be saved.

        The directory may not exist yet.
        """
        return self.base_path.join('webserver', servername, self.app_name)

    def get_feature_path(self, featurename):
        """ Returns the Path, under wich a feature may store it's per-app-files

        The directory is shared between revisions, and therefore is usable
        for database files, etc.

        The directory may not exist yet.
        """
        return self.base_path.join('apps', self.app_name, featurename)


class RevisionEnvironment(object):
    """ Environment of a Revision.

    The things a handler does should use as many of the provides helpers
    as possible

    app_name
        The name of the Application
    revision_name
        The name of the Revision
    base_path
        the root path, where applications reside
    revision_path
        the path where all revision specific data resides
    logger
        callable, to log installation logs
    """
    def __init__(self, base_path, app_name, revision_name, logger):
        self.base_path = Path(base_path)
        self.revision_path = self.base_path.join('apps', app_name,
                                                    'revisions', revision_name)
        self.app_name = app_name
        self.revision_name = revision_name
        self.logger = logger

    def add_config(self, config_name, configdict):
        """ Creates a config file.

        config_name
            name of the configuration
        configdict
            values of the configuration
        """
        # make sure, config path exists
        makedirs(self.revision_path.join('config'))
        # write config data
        config_file = self.revision_path.join('config',
                                              '%s.json' % config_name)
        config_file.content = json.dumps(configdict)

    def get_config(self):
        config = {}
        for filepath in self.revision_path.glob('config/*.json'):
            config[filepath.basename[:-len('.json')]] = json.loads(
                                                              filepath.content)
        return config

    def get_server_log(self):
        """ Returns the content of the server log """
        try:
            return self.revision_path.join('instance.log').content
        except IOError, exc:
            return u'no log available: %r' % exc


class BackupEnvironment(object):
    def __init__(self, path):
        self.path = Path(path)

    def get_feature_path(self, feature_name):
        return self.path.join('features', feature_name)

# circular dependencies suck
from . import general, wsgi, apache, sqlite, mysql, memcached, mediadir
