# coding: utf-8
"""
 :copyright: (c) 2011 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""
from __future__ import with_statement, absolute_import

from jinja2 import Template
from homunculus_server.helpers import ShellCommand
from . import app_removed, revision_added, current_revision_set, \
                          webserver_reload, alias_added, wsgi_application_added


reload_apache = ShellCommand('sudo', '/etc/init.d/apache2', 'reload')


_apache_virtualhost_template = Template("""\
{% for domain in domains %}
<VirtualHost *:80>
  ServerName {{ domain }}
  ServerAlias *.{{ domain }}

  Include "{{ revision_config_path }}"
</VirtualHost>
{% endfor %}
""")


@current_revision_set.connect
def handle_apache_set_current_revision(hom_env, app_env, revision_env):
    """ creates the virtualhost-config for the given revision
    """
    revision_config_path = revision_env.revision_path.join('apache.cfg')

    app_config = app_env.get_virtualhost_path('apache')
    app_config.dirname.makedirs()
    app_config.content = _apache_virtualhost_template.render(
        revision_config_path=revision_config_path,
        domains=app_env.domains,
    )

    webserver_reload.send(hom_env, app_env=app_env, revision_env=revision_env)


@webserver_reload.connect
def handle_apache_reload(hom_env, app_env, revision_env):
    """ reloads the apache
    """
    reload_apache()


@alias_added.connect
def on_alias_added(hom_env, app_env, revision_env, alias, path):
    """ Mounts a static path

    creates the apache.cfg when neccessary.
    """
    config_file = revision_env.revision_path.join('apache.cfg')
    config_file.append('Alias %s %s\n' % (alias, path))


@wsgi_application_added.connect
def on_wsgi_application_added(hom_env, app_env, revision_env, alias, path):
    """Adds a WSGIScriptAlias line in $path/apache.cfg.

    creates the apache.cfg when neccessary.
    """
    config_file = revision_env.revision_path.join('apache.cfg')
    config_file.append('WSGIScriptAlias %s %s\n' % (alias, path))


@revision_added.connect
def on_revision_added_configure_apache_log(hom_env, app_env, revision_env,
                                                                     **kwargs):
    """ Adds the lines for custom logs to $path/apache.cfg.

    creates the apache.cfg when neccessary.
    """
    config_file = revision_env.revision_path.join('apache.cfg')
    config_file.append('CustomLog %s common\n'
                             % revision_env.revision_path.join('instance.log'))
    config_file.append('ErrorLog %s\n'
                             % revision_env.revision_path.join('instance.log'))


@app_removed.connect
def on_app_removed_remove_apache_config(hom_env, app_env):
    """ Removes all apache configuration files for the given app
    """
    app_config = app_env.get_virtualhost_path('apache')
    if app_config.exists:
        app_config.remove()
        reload_apache()
