# coding: utf-8
"""

Hosting a Python WSGI-Application
=================================

Packages can contain python wsgi-applications.
To serve one, add this lines to your features: ::

    "python": {
        "dependencies": [
            "flask",
            "python-mysql",
            # ...
        ],
        "wsgi": {
            "my.controller.module": "/",
            # ...
        }
    }

The controller module given in python.wsgi must contain a variable
`application` that is as WSGI-application.

The given dependencies must be PyPi-names for python libraries. They will be
installed via the fine package manager PIP.


 :copyright: (c) 2011 Philipp Benjamin Köppchen
 :license: GPLv3, see LICENSE for more details.
"""

from __future__ import absolute_import

import homunculus_server
from homunculus_server.helpers import Virtualenv, Path
from . import revision_added, wsgi_application_added, feature_added


RUNNER_WSGI_TEMPLATE = """\
#! /usr/bin/python

# activate the virtualenv
activate_this = %(activate_this)r
execfile(activate_this, {'__file__': activate_this})

# initialize the instancetools
import homunculus_instancetools
homunculus_instancetools.init(%(revision_path)r)

# add application to sys.path
import sys
sys.path.insert(0, %(revision_raw_path)r)

# aliasing homunculus_instancetools as homunculus_config for legacy
sys.modules['homunculus_config'] = homunculus_instancetools

# import application for wsgi usage
from %(wsgi_module)s import application
"""


@feature_added.connect_via(intern('python'))
def create_virtualenv(feature, hom_env, app_env, revision_env, feature_config):
    """Sets up a virtualenv in $path/virt, and installs all dependencies
    """
    revision_env.logger('setting up virtualenv')
    virtualenv = Virtualenv(revision_env.revision_path.join('virt'))
    virtualenv.create()

    revision_env.logger('installing homunculus_instancetools')
    package_path = (Path(homunculus_server.__file__).dirname
                 .join('resources', 'homunculus_instancetools-current.tar.gz'))
    virtualenv.install(package_path)

    for dependency in feature_config.get('dependencies', []):
        revision_env.logger('installing dependency %s' % dependency)
        virtualenv.install(dependency)


@feature_added.connect_via(intern('python'))
def create_wsgi_runner(feature, hom_env, app_env, revision_env,
                                                               feature_config):
    """Creates wsgi runners for every entry in python.wsgi
    """
    mounts = feature_config.get('wsgi', {}).items()

    for index, (wsgi_module, mount_path) in enumerate(mounts):
        revision_env.logger('mounting wsgi-application %s to %s'
                                                   % (wsgi_module, mount_path))

        runner_file = revision_env.revision_path.join('runner_%s.wsgi' % index)
        runner_file.content = RUNNER_WSGI_TEMPLATE % {
            'activate_this': revision_env.revision_path.join('virt', 'bin',
                                                           'activate_this.py'),
            'revision_path': revision_env.revision_path,
            'revision_raw_path': revision_env.revision_path.join('raw'),
            'wsgi_module': wsgi_module,
        }

        wsgi_application_added.send(hom_env, app_env=app_env,
                                             revision_env=revision_env,
                                             alias=mount_path,
                                             path=runner_file)
