import colorconsole, colorconsole.terminal, colorconsole.win
import sys, traceback
from mytools import OptionParser
import os, os.path
import time
import datetime



import threading
try:
    import Queue as queue
except:
    import queue

from mytools.consoleadvance import *
import psutil

VERSION="0.2"
PROG="htopp"
DESCRIPTION="""
htop by python
htop <PARAM>=<VALUE> <PARAM>=<VALUE> ...
list of param:
  - cpu_interval      default 1
  - mem_interval      default 1
  - swap_interval     default 1
  - process_interval  default 0.3
  - process_queue     default NUM_CPUS
  - process_compareby default 'cpu_percent'
  - process_denied    default True
  - refresh_interval  default 2
"""

AUTHOR="Frederic Aoustin" 


CONF = {'color':
            {'text' : 'LPURPLE',
            'value' : 'LGREY',
            'user' : 'LRED',
            'system' : 'LGREEN',
            'delimiter' : 'WHITE',
            'header_bg':'LGREEN',
            'header_fg':'BLACK',
            'procs' : 'WHITE',
            },
         'delimiter':'|',   
       }
COLOR = CONF['color']       

COMPARATOR = ['memory_percent', 
                'cpu_percent',
                'cpu_times', 
                'status'] 

PARAM = {}

def formatArgs(value):
    if value in ('True','true') : return True
    if value in ('False', 'false'): return False
    try:
        try:
            return int(value)
        except:    
            return float(value)
    except:
        return value

def onlyRunning(p):
    if p['status'] == 'R':
        return True
    return False    

def all(p):
    return True

class ListProcess(list, object):

    def __init__(self, lst=[], compareby='cpu_percent', denied = False, interval=None):
        self.compareby = compareby
        self.denied = denied
        self.interval = interval
        for i in lst:
            self.append(lst)

    @property
    def compareby(self):
        return self._compareby

    @compareby.setter
    def compareby(self, value):
        if value not in COMPARATOR:
            raise AttributeError('%s not comparator' % value)
        self._compareby = value
        for i in self:
            i.compareby = value

    @property
    def denied(self):
        return self._denied

    @denied.setter
    def denied(self, value):
        self._denied = value
        for i in self:
            i.denied = value
    
    @property
    def interval(self):
        return self._interval

    @interval.setter
    def interval(self, value):
        self._interval = value
        for i in self:
            i.interval = value

    def append(self, val):
        p = Process(val, self.compareby, self.denied, self.interval)
        super(ListProcess, self).append(p)

    def remove(self, proc):
        super(ListProcess, self).remove(proc)
    
class Process(psutil.Process):
   
    def __init__(self, pid, compareby='cpu_percent', denied= False, interval=None):
        psutil.Process.__init__(self, pid)
        self.interval = interval
        self.denied = denied
        self.update = False
        self.compareby = compareby
        self.refresh()

    @property
    def compareby(self):
        return self._compareby

    @compareby.setter
    def compareby(self, value):
        if value not in COMPARATOR:
            raise AttributeError('%s not comparator' % value)
        self._compareby = value

    def __getitem__(self, name, default=''):
        try:
            return self.container[name]
        except:
            return default

    def get(self, val, default=''):
        return self.__getitem__(val, default)

    def __len__(self):
        return len(self.container)
    
    def keys(self):
        return self.container.keys()

    def __lt__(self, other):
        if self[self.compareby] == other[self.compareby]:
            return self.pid < other.pid  
        return self[self.compareby] < other[self.compareby]

    def __le__(self, other):
        if self[self.compareby] == other[self.compareby]:
            return self.pid <= other.pid  
        return self[self.compareby] <= other[self.compareby]

    def __eq__(self, other):
        return self.pid == other.pid

    def __ne__(self, other):
        return self.pid != other.pid

    def __gt__(self, other):
        if self[self.compareby] == other[self.compareby]:
            return self.pid > other.pid  
        return self[self.compareby] > other[self.compareby]

    def __ge__(self, other):
        if self[self.compareby] == other[self.compareby]:
            return self.pid >= other.pid  
        return self[self.compareby] >= other[self.compareby]

    def refresh(self):
        self.container = self.__get_process_stats()
        self.update = True

    def __get_process_stats(self):
        """
        Get process statistics
        """
        procstat = {}
        procstat['name'] = self.name
        procstat['pid'] = self.pid
        try:
            procstat['username'] = self.username
        except :
            if self.denied == False:
                raise psutil.AccessDenied 
            try:
                procstat['username'] = self.uids.real
            except:
                procstat['username'] = "?"
        procstat['cmdline'] = ' '.join(self.cmdline)
        procstat['memory_info'] = self.get_memory_info()
        procstat['memory_percent'] = self.get_memory_percent()
        procstat['status'] = str(self.status)[:1].upper()
        procstat['cpu_times'] = self.get_cpu_times()
        procstat['cpu_percent'] = self.get_cpu_percent(interval=self.interval)
        procstat['create_time'] = self.create_time
        try:
            procstat['nice'] = self.get_nice()
        except:
            procstat['nice'] = 0

        today_day = datetime.date.today()
        ctime = datetime.datetime.fromtimestamp(procstat['create_time'])
        if ctime.date() == today_day:
            ctime = ctime.strftime("%H:%M")
        else:
            ctime = ctime.strftime("%b%d")
        procstat['ctime'] = ctime
        procstat['cputime'] = time.strftime("%M:%S", time.localtime(sum(procstat['cpu_times'])))
        user = procstat['username']
        if os.name == 'nt' and '\\' in user:
            user = user.split('\\')[1]
        procstat['user'] = user
        procstat['vms'] = procstat['memory_info'] and \
            int(procstat['memory_info'].vms / 1024) or -1
        procstat['rss'] = procstat['memory_info'] and \
            int(procstat['memory_info'].rss / 1024) or -1
        procstat['memp'] = procstat['memory_percent'] and \
            round(procstat['memory_percent'], 1) or -1
        return procstat


class ThreadGen(threading.Thread):
    def __init__(self, server, interval, func, value):
        threading.Thread.__init__(self)
        self.server = server
        self.func = func
        self.value = value
        self.interval = interval
        self.terminated = False
    
    def stop(self):
        self.terminated = True

    def run(self):
        while not self.terminated:
            exec("self.server.%s = self.func()" % self.value)
            time.sleep(self.interval)

class ThreadCpu(threading.Thread):
    def __init__(self, server, interval):
        threading.Thread.__init__(self)
        self.server = server
        self.cpu_times_history = psutil.cpu_times()
        self.interval = interval
        self.terminated = False
    
    def stop(self):
        self.terminated = True

    def run(self):
        while not self.terminated:
            val = psutil.cpu_percent(interval=self.interval, percpu=False) 
            cpu_times = psutil.cpu_times(percpu=True)
            try:
                perc_system = (cpu_times.system - self.cpu_times_history.system) * 100 / ((cpu_times.user - self.cpu_times_history.user) + (cpu_times.system - self.cpu_times_history.system))
                perc_user = (cpu_times.user - self.cpu_times_history.user) * 100 / ((cpu_times.user - self.cpu_times_history.user) + (cpu_times.system - self.cpu_times_history.system))
            except:
                perc_system = 100
                perc_user = 0
            self.cpu_times_history = cpu_times
            self.server.cpu = [val, perc_system, perc_user]


class ThreadPercpu(threading.Thread):
    def __init__(self, server, interval):
        threading.Thread.__init__(self)
        self.server = server
        self.per_cpu_times_history = psutil.cpu_times(percpu=True)
        self.interval = interval
        self.terminated = False
    
    def stop(self):
        self.terminated = True

    def run(self):
        while not self.terminated:
            result = []
            per_val = psutil.cpu_percent(interval=self.interval, percpu=True) 
            per_cpu_times = psutil.cpu_times(percpu=True)
            for val, cpu_times, cpu_times_history in zip(per_val, per_cpu_times, self.per_cpu_times_history):
                try:
                    perc_system = (cpu_times.system - cpu_times_history.system) * 100 / ((cpu_times.user - cpu_times_history.user) + (cpu_times.system - cpu_times_history.system))
                    perc_user = (cpu_times.user - cpu_times_history.user) * 100 / ((cpu_times.user - cpu_times_history.user) + (cpu_times.system - cpu_times_history.system))
                except:
                    perc_system = 100
                    perc_user = 0
                result.append({'val': val, 'system': perc_system, 'user': perc_user})    
            self.per_cpu_times_history = per_cpu_times
            self.server.percpu = result


class ThreadProcessRefresh(threading.Thread):
    def __init__(self, qsrc, qdst, lprocs):
        threading.Thread.__init__(self)
        self.qsrc = qsrc
        self.qdst = qdst
        self.lprocs = lprocs

    def run(self):
        while not self.qsrc.empty():
            pid = self.qsrc.get()
            try:
                self.lprocs.append(pid)
            except Exception as e: #(psutil.NoSuchProcess, psutil.AccessDenied):
                pass
            self.qdst.put(pid)


class ThreadProcess(threading.Thread):
    def __init__(self, server, interval, q, compareby, denied):
        threading.Thread.__init__(self)
        self.server = server
        self.interval = interval
        self.cnt_queue = q 
        self.compareby = compareby
        self.denied = denied
        self.terminated = False
    
    def stop(self):
        self.terminated = True

    def run(self):
        while not self.terminated:
            lprocs = ListProcess(compareby = self.compareby, denied = self.denied, interval=self.interval) 
            qsrc = queue.Queue()
            for i in [i.pid for i in psutil.process_iter()]:
                qsrc.put(i)
            qdst = queue.Queue(qsrc.qsize())
            l = qsrc.qsize()
            for i in range(0, self.cnt_queue):
                ThreadProcessRefresh(qsrc, qdst, lprocs).start()
            while not qdst.full() and not self.terminated:
                time.sleep(self.interval)
            self.server.process = lprocs                        

class ServerHtop(object):

    def __init__(self, conf={}):
        self.NUM_CPUS = psutil.NUM_CPUS
        self.conf = conf
        self._cpu = {'val':0, 'system':0, 'user':0}
        self._percpu = [ {'val':0, 'system':0, 'user':0} for i in range(0,self.NUM_CPUS)]
        self.info = {}
        self.info['cpu'] = self._cpu
        self.info['percpu'] = self._percpu
        self.info['mem'] ={'percent':0, 'total':0, 'available':0, 'used':0} 
        self.info['swap'] = {'percent':0, 'total':0, 'available':0, 'used':0} 
        self.info['process'] = []
        self._servers = {
            'cpu' : ThreadCpu(self, conf.get('cpu_interval',1)),
            'percpu' : ThreadPercpu(self, conf.get('cpu_interval',1)),
            'mem' : ThreadGen(self, conf.get('mem_interval',1), psutil.virtual_memory, 'mem'),
            'swp' : ThreadGen(self, conf.get('swap_interval',1), psutil.swap_memory, 'swap'),
            'process' : ThreadProcess(self, conf.get('process_interval',0.3), conf.get('process_queue',self.NUM_CPUS), 
                    conf.get('process_compareby','cpu_percent'), conf.get('process_denied',True) )
                    }
    
    def start(self):
        for i in self._servers:
            self._servers[i].start()

    def stop(self):
        for i in self._servers:
            self._servers[i].stop()

    @property
    def cpu(self):
        return self.info['cpu']

    @cpu.setter
    def cpu(self, value):
        if len(value) != 3:
            raise AttributeError('value not list [val, system, user]')
        self.info['cpu']['val'] = value[0]
        self.info['cpu']['system'] = value[1]
        self.info['cpu']['user'] = value[2]

    @property
    def percpu(self):
        return self.info['percpu']

    @percpu.setter
    def percpu(self, value):
        if len(value) != self.NUM_CPUS:
            raise AttributeError('number of cpu is wrong')
        self.info['percpu'] = value

    @property
    def mem(self):
        return self.info['mem']

    @mem.setter
    def mem(self, value):
        self.info['mem']['percent'] =  value.percent
        self.info['mem']['total'] = value.total
        self.info['mem']['available'] = value.available
        self.info['mem']['used'] = value.total - value.available

    @property
    def swap(self):
        return self.info['swap']

    @swap.setter
    def swap(self, value):
        self.info['swap']['percent'] =  value.percent
        self.info['swap']['total'] = value.total
        self.info['swap']['used'] = value.used
        self.info['swap']['available'] = value.total - value.used

    @property
    def process(self):
        return self.info['process']

    @process.setter
    def process(self, value):
        self.info['process'] = value

    @property
    def uptime(self):
        return str(datetime.datetime.now() - datetime.datetime.fromtimestamp(psutil.BOOT_TIME))[:-7]

    def cnt_process(self, filtered=all):
        return len(list(filter(filtered, self.info['process'])))


class Htopp(Screen):

    def __init__(self, term, server = None):
        Screen.__init__(self, term)
        self.server = server
        #self.set_title(PROG)
        self.clear()
        self.reset()
        self.no_cursor()
        self._init_screen()
        self._init_bind()
        self.lprocs = ListProcess(compareby = 'cpu_percent', denied = True)   

    def _init_screen(self):
        x = 3
        y = 1
        ln = int(self.width /2) -1
        self.ln_progressbar = ln -x
        lst = ['Cpu'] + list(range(1,self.server.NUM_CPUS+1,1)) + ['Mem','Swp']
        self.pbs = []
        self.procs = []
        #manage ProgressBar
        for i in lst:
            pb = []
            Text(self, x+0, y, l=3, fg=COLOR['text']).value = str(i)
            Text(self, x+3, y, l=1, fg=COLOR['delimiter']).value = '['
            Text(self, x+ln, y, l=1, fg=COLOR['delimiter']).value = ']'
            for i in range(1, self.ln_progressbar,  1):
                a = Text(self,x+3+i, y, l=1, fg=COLOR['user'])
                a.value = ' '
                pb.append(a)
            self.pbs.append(pb)    
            y = y + 1
        y = y + 1
        self.templ ="%-10s %5s %5s %4s %7s %7s %-13s %5s %7s  %s"
        self.cnt_process = self.height - y    
        self.ln_process = self.width
        
        header = self.templ %("USER", "PID", "%CPU", "%MEM", "VSZ", "RSS", "TTY", "START","TIME", "COMMAND")
        Text(self, 0, y, l=self.ln_process, fg = COLOR['header_fg'],bg=COLOR['header_bg']).value = header
        y = y + 1
        for i in range(1,self.cnt_process,1):
           self.procs.append(Text(self, 0, y, l=self.ln_process, fg = COLOR['procs']))
           y = y +1 
        
        Text(self, ln+2*x, 1, l=9, fg=COLOR['text']).value = 'Uptime : '
        self.text_uptime = Text(self, ln+2*x+9, 1, l=20, fg=COLOR['value'])
 
        Text(self, ln+2*x, 2, l=9, fg=COLOR['text']).value = 'Process: '
        self.text_all_process = Text(self, ln+2*x+9, 2, l=4, fg=COLOR['value'])
       
        Text(self, ln+2*x, 3, l=9, fg=COLOR['text']).value = 'Running: '
        self.text_run_process = Text(self, ln+2*x+9, 3, l=4, fg=COLOR['value'])
        
    def _init_bind(self):
        #self.start_listen()
        pass

    def refresh(self):
        #pass
        #refresh cpu
        for perc,cpu in zip([self.server.cpu],self.pbs):
            color_user, color_val = self.get_dashes(perc['val'], self.ln_progressbar, [COLOR['user'],],[COLOR['value'],])
            try:
                color_system, color_user = self.get_dashes( perc['system'], len(color_user), [COLOR['system'],], [COLOR['user'],])
            except:
                color_system = []
            color_all = color_system + color_user + color_val
            d, e = self.get_dashes(perc['val'], self.ln_progressbar, str(CONF['delimiter']), ' ')
            v = (d + e)[:-7] + '{:>5}%'.format(perc['val'])
            for text, val, color in zip(cpu, v, color_all):
                text.fg = color
                text.value = val
        num = 1        
        for perc,cpu in zip(self.server.percpu,self.pbs[1:]):
            color_user , color_val = self.get_dashes(perc['val'], self.ln_progressbar, [COLOR['user'],],[COLOR['value'],])
            try:
                color_system, color_user = self.get_dashes( perc['system'], len(color_user), [COLOR['system'],], [COLOR['user'],])
            except:
                color_system = []
            color_all = color_system + color_user + color_val
            d, e = self.get_dashes(perc['val'], self.ln_progressbar, str(CONF['delimiter']), ' ')
            v = (d + e)[:-7] + '{:>5}%'.format(perc['val'])
            for text, val, color in zip(cpu, v, color_all):
                text.fg = color
                text.value = val
            num = num +1    
        #refresh mem
        mem = self.server.mem
        perc = mem['percent']
        color_user , color_val = self.get_dashes(perc, self.ln_progressbar, [COLOR['user'],],[COLOR['value'],])
        color_all = color_user + color_val
        d, e = self.get_dashes(perc, self.ln_progressbar, str(CONF['delimiter']), ' ')
        txt =str(int((mem['used']) / 1024 / 1024)) + "/" + str(int(mem['total'] / 1024 / 1024)) + "MB" 
        v = (d + e)[:-1*(len(txt)+1)] + txt
        for text, val, color in zip(self.pbs[-2], v, color_all):
            text.fg = color
            text.value = val
        #refresh swp
        mem = self.server.swap
        perc = mem['percent']
        color_user , color_val = self.get_dashes(perc, self.ln_progressbar, [COLOR['user'],],[COLOR['value'],])
        color_all = color_user + color_val
        d, e = self.get_dashes(perc, self.ln_progressbar, str(CONF['delimiter']), ' ')
        txt =str(int(mem['used'] / 1024 / 1024)) + "/" + str(int(mem['total'] / 1024 / 1024)) + "MB" 
        v = (d + e)[:-1*(len(txt)+1)] + txt
        for text, val, color in zip(self.pbs[-1], v, color_all):
            text.fg = color
            text.value = val
        #refresh procs
        k = 0
        self.lprocs = self.server.process
        self.lprocs.sort()
        self.lprocs.reverse()
        while k < (self.cnt_process -1) and len(self.procs)>0:
            j = self.procs[k]
            if k < len(self.lprocs):
                i = self.lprocs[k]    
                j.value = self.templ % (i['user'][:10],
                            i['pid'],
                            i['cpu_percent'],
                            i['memp'],
                            i['vms'],
                            i['rss'],
                            i.get('terminal', '') or '?',
                            i['ctime'],
                            i['cputime'],
                            i['name'].strip() or '?')
            else:
                j.value = ''
            k = k +1  
        self.text_all_process.value = self.server.cnt_process()
        self.text_run_process.value = self.server.cnt_process(onlyRunning)
        self.text_uptime.value = self.server.uptime
    
    def get_dashes(self, perc, ln, check, nocheck):
        dashes = check * int(perc * ln / 100)
        empty_dashes = nocheck * (ln - len(dashes))
        return dashes , empty_dashes



def load():
    usage = "usage: %prog [options]" 
    parser = OptionParser(version="%s %s" % (PROG,VERSION), usage=usage)

    parser.description= '\n'.join([ parser.get_formatwidth() % i for i in DESCRIPTION.split('\n')])
    parser.epilog = AUTHOR
    try:
        (options, args) = parser.parse_args()
        myconf = {}
        for i in args:
            if len(i.split('=')) == 2:
                myconf[i.split('=')[0]] = formatArgs(i.split('=')[1])
        server = ServerHtop(conf=myconf)
        server.start()
        screen = Htopp(parser.screen, server)
        try:
            while(True):
                screen.refresh()
                time.sleep(myconf.get('refresh_interval',2))
        except KeyboardInterrupt as e:
            server.stop()
        screen.clear()
        screen.reset()
        sys.exit()
    except Exception as e:
        exc_type, exc_value, exc_traceback = sys.exc_info()
        formatted_lines = traceback.format_exc().splitlines()
        parser.print_stdout('\n'.join(formatted_lines),'LRED','BLACK')
        parser.error(e)
        sys.exit(1)     

# manage filter or order
# ajout thread reseau ?
