import unittest
import random
import string
import hyperestraier

def getUid():
    random_string = ''
    for i in range(16):  
        random_string += random.choice(string.letters + string.digits)
    return random_string 


class NodeTestCase(unittest.TestCase):
    
    def setUp(self):
        self.node = hyperestraier.Node()
        self.node.set_url("http://localhost:1978/node/test")
        self.node.set_auth("admin", "admin")
    
    def tearDown(self):
        pass

    def testPutDoc(self):
        doc = hyperestraier.Document()
        uid = getUid()
        doc.add_attr("@uri", "http://localhost/" + uid)
        doc.add_attr("@title", "Over the rainbow")
        doc.add_text("There's a land that I heard of once in a lullaby.")
        doc.add_text("Somewhere over the rainbow.  Way up high.")
        result = self.node.put_doc(doc)
        self.assertEqual(result, True)

    def testPutDocErr(self):
        doc = hyperestraier.Document()
        uid = getUid()
        doc.add_attr("@title", "Over the rainbow")
        doc.add_text("There's a land that I heard of once in a lullaby.")
        doc.add_text("Somewhere over the rainbow.  Way up high.")
        self.assertRaises(hyperestraier.HyperestraierError, self.node.put_doc, doc)

    def testSync(self):
        result = self.node.sync()
        self.assertEqual(result, True)

    def testOptimize(self):
        result = self.node.optimize()
        self.assertEqual(result, True)

    def testOutDoc(self):
        uid = getUid()
        doc = hyperestraier.Document()
        doc.add_attr("@uri", "http://localhost/"+uid)
        doc.add_attr("@title", "Over the rainbow")
        doc.add_text("There's a land that I heard of once in a lullaby.")
        doc.add_text("Somewhere over the rainbow.  Way up high.")
        result = self.node.put_doc(doc)
        self.assertEqual(result, True)
        
        hid = self.node.uri_to_id("http://localhost/"+uid)
        self.assertNotEqual(hid, None)
        
        result = self.node.out_doc(hid)
        self.assertEqual(result, True)

        doc = hyperestraier.Document()
        uid = getUid()
        doc.add_attr("@uri", "http://localhost/"+uid)
        doc.add_attr("@title", "Over the rainbow")
        doc.add_text("There's a land that I heard of once in a lullaby.")
        doc.add_text("Somewhere over the rainbow.  Way up high.")
        result = self.node.put_doc(doc)
        self.assertEqual(result, True)
        
        result = self.node.out_doc_by_uri("http://localhost/"+uid)
        self.assertEqual(result, True)

    def testEditDoc(self):
        uid = getUid()
        doc = hyperestraier.Document()
        doc.add_attr("@uri", "http://localhost/"+uid)
        doc.add_attr("@title", "Over the rainbow")
        doc.add_text("There's a land that I heard of once in a lullaby.")
        doc.add_text("Somewhere over the rainbow.  Way up high.")
        result = self.node.put_doc(doc)
        self.assertEqual(result, True)

        doc = self.node.get_doc_by_uri("http://localhost/"+uid)
        result = self.node.edit_doc(doc)
        self.assertEqual(result, True)

    def testGetDoc(self):
        uid = getUid()
        doc = hyperestraier.Document()
        doc.add_attr("@uri", "http://localhost/"+uid)
        doc.add_attr("@title", "Over the rainbow")
        doc.add_text("There's a land that I heard of once in a lullaby.")
        doc.add_text("Somewhere over the rainbow.  Way up high.")
        result = self.node.put_doc(doc)
        self.assertEqual(result, True)
        
        result = self.node.get_doc_by_uri("http://localhost/"+uid)
        self.assertNotEqual(result, None)
        self.assertEqual(result.attr("@title"), "Over the rainbow")
        self.assertEqual(result.attr("@uri"), "http://localhost/"+uid)

        hid = self.node.uri_to_id("http://localhost/"+uid)
        self.assertNotEqual(hid, None)

        result = self.node.get_doc(hid)
        self.assertNotEqual(result, None)
        self.assertEqual(result.attr("@title"), "Over the rainbow")
        self.assertEqual(result.attr("@uri"), "http://localhost/"+uid)

    def testAttr(self):
        uid = getUid()
        doc = hyperestraier.Document()
        doc.add_attr("@uri", "http://localhost/"+uid)
        doc.add_attr("@title", "Over the rainbow")
        doc.add_text("There's a land that I heard of once in a lullaby.")
        doc.add_text("Somewhere over the rainbow.  Way up high.")
        result = self.node.put_doc(doc)
        self.assertEqual(result, True)
        
        hid = self.node.uri_to_id("http://localhost/"+uid)
        self.assertNotEqual(hid, None)
        
        result = self.node.get_doc_attr(hid, "@title")
        self.assertEqual(result, "Over the rainbow")
        result = self.node.get_doc_attr(hid, "@uri")
        self.assertEqual(result, "http://localhost/"+uid)

        uri = "http://localhost/"+uid
        result = self.node.get_doc_attr_by_uri(uri, "@title")
        self.assertEqual(result, "Over the rainbow")
        result = self.node.get_doc_attr_by_uri(uri, "@uri")
        self.assertEqual(result, "http://localhost/"+uid)

    def testKword(self):
        uid = getUid()
        doc = hyperestraier.Document()
        doc.add_attr("@uri", "http://localhost/"+uid)
        doc.add_attr("@title", "Over the rainbow")
        doc.add_text("There's a land that I heard of once in a lullaby.")
        doc.add_text("Somewhere over the rainbow.  Way up high.")
        doc.set_keywords({"high": "10.0", "there": "20.0"})
        result = self.node.put_doc(doc)
        self.assertEqual(result, True)
        
        hid = self.node.uri_to_id("http://localhost/"+uid)
        self.assertNotEqual(hid, None)
        
        result = self.node.etch_doc(hid)
        self.assertNotEqual(result, None)

        result = self.node.etch_doc_by_uri("http://localhost/"+uid)
        self.assertNotEqual(result, None)

    def testSearch(self):
        uid = getUid()
        doc = hyperestraier.Document()
        doc.add_attr("@uri", "http://localhost/"+uid)
        doc.add_attr("@title", "Over the rainbow")
        doc.add_text("There's a land that I heard of once in a lullaby.")
        doc.add_text("Somewhere over the rainbow.  Way up high.")
        doc.set_keywords({"high": "10.0", "there": "20.0"})
        result = self.node.put_doc(doc)
        self.assertEqual(result, True)
        
        cond = hyperestraier.Condition()
        cond.set_phrase("rainbow AND lullaby")
        nres = self.node.search(cond, 0)
        self.assertNotEqual(len(nres.docs), 0)
        
    def testNode(self):
        print self.node.get_name()
        print self.node.get_label()
        print self.node.get_doc_num()
        print self.node.get_word_num()
        print self.node.get_size()
        print self.node.get_cache_usage()
        print self.node.get_admins()
        print self.node.get_users()
        print self.node.get_links()

if __name__ == "__main__":
    unittest.main()
