# Copyright (c) 2013 Jason Ish
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
# STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
# IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

from __future__ import print_function

import sys

class MsgMapEntry(object):
    """ This class represents an entry in the MsgMap. """

    def __init__(self, gid, sid, msg, refs=[]):
        self.gid = gid
        self.sid = sid
        self.msg = msg
        self.refs = refs

    def __repr__(self):
        return "%d || %d || %s || %s" % (
            self.gid, self.sid, self.msg, ",".join(self.refs))

class MsgMap(object):
    """ This class implements of map of event descriptions useful for
    looking up an event description by its gid and sid.
    """

    def __init__(self):
        self.map = {}

    def get(self, gid, sid):
        """ Get a MsgMapEntry by gid and sid.

        A special case is made for gid 3, if an entry is found it will
        be checked for again with gid 1.
        """

        entry_id = (gid, sid)
        if entry_id in self.map:
            return self.map[entry_id]
        
        # Try to do the best thing.  If the gid is 3, try again with
        # gid 1.
        if gid == 3:
            return self.get(1, sid)

        return None

    def add_entry(self, entry):
        """ A a MsgMapEntry to the map. """
        self.map[(entry.gid, entry.sid)] = entry

    def load_sidmsg_file(self, filename, gid=1):
        """ Load a sid-msg.map style file as provided by a Snort
        ruleset or generated by pulledpork. """
        with open(filename) as fileobj:
            for line in fileobj:
                line = line.strip()
                if not line or line.startswith("#"):
                    continue
                parts = [part.strip() for part in line.split("||")]
                entry = MsgMapEntry(gid, int(parts[0]), parts[1], parts[2:])
                self.add_entry(entry)

    def load_genmsg_file(self, filename):
        """ Load a gen-msg.map style file as provided by a Snort
        ruleset. """
        with open(filename) as fileobj:
            for line in fileobj:
                line = line.strip()
                if not line or line.startswith("#"):
                    continue
                gid, sid, msg = [part.strip() for part in line.split("||")]
                entry = MsgMapEntry(gid, sid, msg, None)
                self.add_entry(entry)

class Classification(object):
    """ A class to represent a Snort rule classification. """

    def __init__(self, name, description, priority):
        self.name = name
        self.description = description
        self.priority = priority

    def __repr__(self):
        return "name=%s; description=%s; priority=%d" % (
            self.name, self.description, self.priority)

class ClassificationMap(object):
    """ A map of classification IDs to a classification for retrieval
    of a classification name, description and priority by ID. """

    def __init__(self):
        self.map = []

    def get_by_id(self, id):
        """ Get a classification by ID. """
        try:
            return self.map[id-1]
        except:
            return None

    def load_classification_file(self, filename):
        """ Load classifications from a Snort style
        classification.config. """
        prefix = "config classification:"
        with open(filename) as fileobj:
            for line in fileobj:
                line = line.strip()
                if line.startswith(prefix):
                    name, description, priority = [
                        part.strip() for part in line[len(prefix):].split(",")]
                    entry = Classification(name, description, int(priority))
                    self.map.append(entry)

    def dump(self, fileobj=sys.stdout):
        """ Dump the contents of the classification map in a human
        readable way. """
        for i in range(len(self.map)):
            entry = self.map[i]
            print("%3d. %s" % (i + 1, entry), file=fileobj)
