# -*- coding:utf-8 -*-
from functools import wraps

class ObjectBuilder(object):
    '''
    Incrementally builds an object from JSON parser events. Events are passed
    into the `event` function that accepts two parameters: event type and
    value. The object being built is available at any time from the `value`
    attribute.

    Example:

        from StringIO import StringIO
        from ijson.parse import basic_parse
        from ijson.utils import ObjectBuilder

        builder = ObjectBuilder()
        f = StringIO('{"key": "value"})
        for event, value in basic_parse(f):
            builder.event(event, value)
        print builder.value

    '''
    def __init__(self):
        def initial_set(value):
            self.value = value
        self.containers = [initial_set]

    def event(self, event, value):
        if event == 'map_key':
            self.key = value
        elif event == 'start_map':
            map = {}
            self.containers[-1](map)
            def setter(value):
                map[self.key] = value
            self.containers.append(setter)
        elif event == 'start_array':
            array = []
            self.containers[-1](array)
            self.containers.append(array.append)
        elif event == 'end_array' or event == 'end_map':
            self.containers.pop()
        else:
            self.containers[-1](value)

def coroutine(func):
    '''
    Wraps a generator which intended to be used as a pure coroutine by
    .send()ing it values. The only thing that the wrapper does is calling
    .next() for the first time which is required by Python generator protocol.
    '''
    @wraps(func)
    def wrapper(*args, **kwargs):
        g = func(*args, **kwargs)
        g.next()
        return g
    return wrapper

@coroutine
def foreach(coroutine_func):
    '''
    Dispatches each JSON array item to a handler coroutine. A coroutine is
    created anew for each item by calling `coroutine_func` callable. The
    resulting coroutine should accept value in the form of tuple of values
    generated by rich JSON parser: (prefix, event, value).

    First event received by foreach should be a "start_array" event.
    '''
    g = None
    base, event, value = yield
    if event != 'start_array':
        raise Exception('foreach requires "start_array" as the first event, got %s' % repr((base, event, value)))
    START_EVENTS = set(['start_map', 'start_array', 'null', 'boolean', 'number', 'string'])
    while True:
        prefix, event, value = yield
        if prefix == base + '.item' and event in START_EVENTS:
            g = coroutine_func()
        if (prefix, event) != (base, 'end_array'):
            g.send((prefix, event, value))

@coroutine
def dispatcher(targets):
    '''
    Dispatches JSON parser events into several handlers depending on event
    prefixes.

    Accepts a list of tuples (base_prefix, coroutine). A coroutine then
    receives all the events with prefixes starting with its base_prefix.
    '''
    while True:
        prefix, event, value = yield
        for base, target in targets:
            if prefix.startswith(base):
                target.send((prefix, event, value))
                break
