import os
import time
import urllib2

from django.db.models import get_models
from django.db.models.fields.files import ImageField

def images_for_model(model):
    """ Generates a list of tuples (path, url) for
    images linked to a model.
    """
    try:
        fields = model._meta.fields
    except AttributeError:
        # Maybe that was not a real model in fact ...
        return []

    img_fields = [field.attname for field in fields
                  if isinstance(field, ImageField)]

    images = []
    for obj in model.objects.all():
        for img_field in img_fields:
            img = getattr(obj, img_field)
            try:
                images.append((img.path, img.url))
            except ValueError:
                # Sometimes calling img.path
                # raises a value error, need to work on that
                pass

    return images

def filter_unexisting_images(images):
    """ Filter the images generated by images_for_model
    to only keep images for which the path does not exist
    locally.
    """
    return [img for img in images
            if not os.path.exists(img[0])]

def retrieve_image(url, path):
    """ Downloads the image.
    """
    f = urllib2.urlopen(url)
    output = open(path, 'wb')
    output.write(f.read())
    output.close()


def sync_images(local_url, remote_url, pause = 10, verbose = False):
    """ Retrieves missing images in your local instance.
    local_url is the way you access the site locally ('http://localhost:8000')
    remote_url is the site containing the images ('http://example.com')
    pause is the wait time (in milliseconds) between fetching two images
    to avoid overload of the site.
    if verbose is set to true, a message will be printed for each image retrieved.
    """
    pause = float(pause/1000)

    models = get_models()
    if verbose:
        print 'Found %s models in the system' % (len(models))

    # Then we compute the list of mising images.
    images = []
    for model in models:
        images.extend(filter_unexisting_images(images_for_model(model)))

    # We might again have some double images.
    images = list(set(images))
    if verbose:
        print 'Found %s images that need to be downloaded' % (len(images))


    # Now we download them.
    for img_path, img_url in images:       
        img_url = img_url.replace(local_url, remote_url)
        if verbose:
            print 'Downloading %s to %s' % (img_url,
                                            img_path)

        retrieve_image(img_url, img_path)
        if verbose:
            print 'Image downloaded'
        
        time.sleep(pause)

    print '%s images retrieved' % (len(images))
