
from urllib import urlencode
from urlparse import urljoin
from urllib2 import urlopen, Request
from json import loads, dumps
import requests

from .exceptions import InstantLabApiException

class InstantLabApiClient(object):
    """
    API client for the InstantLab REST API.

    Names are according to
    http://en.wikipedia.org/wiki/URI_scheme#Generic_syntax
    """

    def __init__(self, base_url, frontend_id, api_key):

        if not base_url.endswith("/"):
            base_url += "/"
        self.base_url = base_url

        self.frontend_id = frontend_id
        self.api_key = api_key


    def _get_api_request_headers(self, user_id):
        """
        Returns the default headers for requests to the API as a dict.
        """
        return {
            'X-APIKey': self.api_key,
            'X-Frontend-ID': self.frontend_id,
            'X-User-ID':user_id
        }

    def _get_api_request(self, path, user_id, *args, **kwargs):
        """
        Builds a request object to query the InstantLab API.

        Basically, it proxies the constructor of ``urllib2.Request``
        and sets a few defaults.

        :param path: The path as string to request to at the API.
        :param *args: Handed to ``urllib2.Request``
        :param **kwargs: Handed to ``urllib2.Request``, some default
                         headers will be set
        """
        kwargs.setdefault('headers', self._get_api_request_headers(user_id))
        url = urljoin(self.base_url, path)
        return Request(url, *args, **kwargs)

    def _api_get(self, path, user_id):
        """
        Issues a GET request to the API with the given path.

        :param path: The path as string to GET request to.
        """
        request = self._get_api_request(path, user_id)
        return urlopen(request)

    def _api_post(self, path,user_id, data={}):
        """
        Issues a POST request to the API with the given path and
        data as JSON.

        :param path: The path as string to POST the data to.
        :param data: A dict containing the data to post.
        """
        encoded_data = urlencode(data)
        request = self._get_api_request(path,user_id, encoded_data)
        return urlopen(request)

    def _api_delete(self, path, user_id):
        """
        Issues a DELETE request to the give path.

        :param path: The path as a string request a DELETE to.
        """
        request = self._get_api_request(path, user_id)
        request.get_method = lambda: 'DELETE'
        return urlopen(request)

    def list_infrastructure_layouts(self, user_id):
        response = self._api_get('infrastructure_layout/', user_id)

        if response.code != 200:
            raise InstantLabApiException(repr(response))

        return loads(response.read())

    def list_infrastructures(self,user_id):
        response = self._api_get('infrastructure/',user_id)

        if response.code != 200:
            raise InstantLabApiException(repr(response))

        return loads(response.read())

    def get_activities(self,user_id):
        response = self._api_get('activity_log/',user_id)

        if response.code != 200:
            raise InstantLabApiException(repr(response),user_id)

        return loads(response.read())

    def create_infrastructure(self, infrastructure_layout_id, user_id):
        """
        Creates an infrastructure from a layout and returns its location.
        """
        request_data = {
            'infrastructure_layout_id': infrastructure_layout_id
        }
        response = self._api_post(
            'infrastructure/', user_id ,request_data
        )

        if response.code != 201:
            raise InstantLabApiException(repr(response))

        return response.info()['Location']


    def create_and_get_infrastructure(self,layout_id,user_id,*args, **kwargs):
        """
        Convenience method to create and receive an infrastructure
        created from a layout.
        """
        location = self.create_infrastructure(layout_id,user_id,*args, **kwargs)

        response = self._api_get(location,user_id)

        if response.code != 200:
            raise InstantLabApiException(repr(response))

        return loads(response.read())

    def get_infrastructure(self, infrastructure_id,user_id):
        response = self._api_get(
            'infrastructure/%s' % infrastructure_id,user_id)

        if response.code != 200:
            raise InstantLabApiException(repr(response))

        return loads(response.read())

    def delete_infrastructure(self, infrastructure_id,user_id):
        response = self._api_delete(
            'infrastructure/%s' % infrastructure_id,user_id
        )

        if response.code != 200:
            raise InstantLabApiException(repr(response))

    def get_vnc_token(self, node_cloud_id, user_id):
        s = requests.Session()
        login_url = 'http://experiments.instantlab.org/sunstone/login'
        login_headers = {'Authorization': 'Basic b25lYWRtaW46dWItaWYtb2stb2ItaWYtYXdi'}
        s.post(login_url, headers=login_headers)

        get_vnc_token_url = 'http://experiments.instantlab.org/sunstone/vm/' + \
            str(node_cloud_id) + '/startvnc'
        r = s.post(get_vnc_token_url)
        if r.status_code == 200:
            return r.json()['token']
        return ""
