"""
The main modul of the iptdump package.
"""
__version__ = '1b1.3'

import ipaddr
import time

IPTABLE_V4 = 4
IPTABLE_V6 = 6
IPTABLE_ALL_ = (IPTABLE_V4, IPTABLE_V6)

TABLE_FILTER = 'filter'
TABLE_NAT = 'nat'
TABLE_MANGLE = 'mangle'
TABLE_RAW = 'raw'
TABLE_ALL_ = (TABLE_NAT, TABLE_FILTER, TABLE_MANGLE, TABLE_RAW)

CHAIN_INPUT = 'INPUT'
CHAIN_PREROUTING = 'PREROUTING'
CHAIN_FORWARD = 'FORWARD'
CHAIN_POSTROUTING = 'POSTROUTING'
CHAIN_OUTPUT = 'OUTPUT'
CHAIN_ALL_ = (CHAIN_INPUT, CHAIN_PREROUTING, CHAIN_FORWARD, CHAIN_POSTROUTING, CHAIN_OUTPUT)

PROTO_UDP = 'udp'
PROTO_TCP = 'tcp'
PROTO_ICMP = 'icmp'
PROTO_ICMPV6 = 'icmpv6'
PROTO_41 = 41
PROTO_47 = 47
PROTO_50 = 50
PROTO_51 = 51
PROTO_89 = 89
PROTO_ALL_ = (PROTO_UDP, PROTO_TCP, PROTO_ICMP, PROTO_ICMPV6, PROTO_41, PROTO_47, PROTO_50, PROTO_51, PROTO_89)

TARGET_ACCEPT = 'ACCEPT'
TARGET_DROP = 'DROP'
TARGET_QUEUE = 'QUEUE'
TARGET_RETURN = 'RETURN'
TARGET_ALL_ = (TARGET_ACCEPT, TARGET_DROP, TARGET_QUEUE, TARGET_RETURN)

class IPVersion:
	"""
	Is a IP version.
	"""
	def __init__(self, ipv):
		"""
		x.__init__(ipv)

		Constructor.
		IP version ipv
		"""
		assert ipv in IPTABLE_ALL_
		self._ipv = ipv

	def getIPversion(self):
		"""
		x.getIPversion() -> int

		Return the IP version.
		"""
		return self._ipv

class IptablesAbstract(IPVersion, dict):
	"""
	Is a dict of tables.
	"""
	def __init__(self, ipv):
		"""
		x.__init__(ipv)

		Constructor.
		IP version ipv
		"""
		IPVersion.__init__(self, ipv)

	def table(self, name):
		"""
		x.table(name) -> object

		If the table object name not know, then create it.
		Return the table object name.
		"""
		assert name in TABLE_ALL_
		if name not in self:
			self.__createTable__(name)
		return self[name]

	def __createTable__(self, table):
		"""
		x.__createTable__(table) <==> x[table] = Table(table, x.getIPversion())
		"""
		self.__setitem__(table, Table(table, self.getIPversion()))

	def dump(self, reduce=False):
		"""
		x.dump([reduce]) -> str

		Return the iptable in the iptables-save form.
		"""
		result = []
		for tablename in TABLE_ALL_:
			if tablename in self:
				result += [
					"# Generated by python-iptdump v%s on %s" % (__version__, time.asctime()),
					"*%s" % tablename
				]
				for chainname in CHAIN_ALL_:
					if chainname in self[tablename]:
						result.append(":%s %s [0:0]" % (chainname, self[tablename][chainname].dumpTarget()))
				for chainname in CHAIN_ALL_:
					if chainname in self[tablename]:
						result.append(self[tablename][chainname].dump(reduce))
				result += ("COMMIT", "# Completed on %s" % time.asctime())
		return "\n".join(result)

class Targetable:
	"""
	This class provide handling of targets.
	"""
	def __init__(self):
		"""
		x.__init__()

		Constructor.
		"""
		self.__target = TARGET_ACCEPT

	def target(self, target):
		"""
		x.target(target) -> self

		Set the target target.
		"""
		assert type(target) == str or self.getIPversion() in target.provideIPVersion()
		self.__target = target
		return self

	def dumpTarget(self):
		"""
		x.dumpTarget() -> str

		Return a str of the target.
		"""
		return str(self.__target)

class Iptables(IptablesAbstract):
	"""
	Iptable object for IP version 4.
	"""
	def __init__(self):
		"""
		x.__init__()

		Constructor.
		"""
		IptablesAbstract.__init__(self, IPTABLE_V4)

class Ip6tables(IptablesAbstract):
	"""
	Iptable object for IP version 6.
	"""
	def __init__(self):
		"""
		x.__init__()

		Constructor.
		"""
		IptablesAbstract.__init__(self, IPTABLE_V6)

class Table(IPVersion, dict):
	"""
	Is a dict of chain objects.
	"""
	def __init__(self, table, ipv):
		"""
		x.__init__(table, ipv)

		Constructor.
		table name table
		IP version ipv
		"""
		IPVersion.__init__(self, ipv)
		self.__name = table

	def __getAllowChainTuple__(self):
		"""
		x.__getAllowChainTuple__() -> tuple

		Return a tupel of allow chains for this table.
		"""
		if self.__name == TABLE_FILTER:
			return (CHAIN_INPUT, CHAIN_FORWARD, CHAIN_OUTPUT)
		elif self.__name == TABLE_NAT:
			return (CHAIN_PREROUTING, CHAIN_POSTROUTING, CHAIN_OUTPUT)
		elif self.__name == TABLE_MANGLE:
			return (CHAIN_INPUT, CHAIN_PREROUTING, CHAIN_FORWARD, CHAIN_POSTROUTING, CHAIN_OUTPUT)
		elif self.__name == TABLE_RAW:
			return (CHAIN_PREROUTING, CHAIN_OUTPUT)

	def chain(self, name):
		"""
		x.chain(name) -> object

		If the chain object name not know, create it.
		Return the chain object name.
		"""
		if name not in self:
			assert name in self.__getAllowChainTuple__()
			self.__createChain__(name)
		return self[name]

	def __createChain__(self, name):
		"""
		x.__createChain__(name) <==> x[name] = Chain(name, x.getIPversion())
		"""
		self.__setitem__(name, Chain(name, self.getIPversion()))

class Chain(IPVersion, Targetable, list):
	"""
	Is a list of rule objects.
	"""
	@staticmethod
	def reduce(rules):
		"""
		Chain.reduce(rules) -> list

		This static methode reduce an list of rules.
		It make each line uniqe and return this.
		"""
		result = []
		for rule in rules:
			if rule not in result:
				result.append(rule)
		return result

	def __init__(self, chain, ipv):
		"""
		x.__init__(chain, ipv)

		Constructor.
		chain name chain
		IP version ipv
		"""
		IPVersion.__init__(self, ipv)
		Targetable.__init__(self)
		self.__name = chain

	def createRule(self):
		"""
		x.createRule() -> object

		Create a new rule, append this internal and return this.
		"""
		r = Rule(self.__name, self.getIPversion())
		self.append(r)
		return r

	def new(self):
		"""
		x.new() -> object

		Alias for createRule().
		"""
		return self.createRule()

	def target(self, target):
		"""
		x.target(target) -> object

		Set the target target and return this.
		"""
		assert target in TARGET_ALL_
		return Targetable.target(self, target)

	def dump(self, reduce=False):
		"""
		x.dump([reduce]) -> str

		Return the rule in a format like iptable-save.
		If reduce true, then reduce the return string of uniqe lines.
		"""
		result = [rule.dump() for rule in self]
		if reduce:
			result = Chain.reduce(result)
		return "\n".join(result)

class Rule(IPVersion, Targetable):
	"""
	Is a iptable rule.
	"""
	def __init__(self, chain, ipv):
		"""
		x.__init__(chain, ipv)

		Constructor.
		chain name chain
		IP version ipv
		"""
		IPVersion.__init__(self, ipv)
		Targetable.__init__(self)
		self.__chain = chain
		self.__extension = []
		self.__proto = None
		self.__iIface = None
		self.__oIface = None
		self.__src = None
		self.__srcInvert = None
		self.__dst = None
		self.__dstInvert = None

	def addExtension(self, ext):
		"""
		x.addExtension(ext) -> self

		Add the extension ext.
		Allowed types of ext is a list or tupe of extension objects or a extension object.
		"""
		if type(ext) in (list, tuple):
			map(self.addExtension, ext)
		else:
			assert self.__chain in ext.provideChain()
			self.__extension.append(ext)
		return self

	def addExt(self, ext):
		"""
		x.addExt(ext) -> self

		Alias for addExtension(ext).
		"""
		return self.addExtension(ext)

	def iIface(self, iface):
		"""
		x.iIface(iface) -> self

		Set the input interface iface.
		"""
		assert self.__chain in (CHAIN_INPUT, CHAIN_PREROUTING, CHAIN_FORWARD)
		self.__iIface = iface
		return self

	def oIface(self, iface):
		"""
		x.oIface(iface) -> self

		Set the output interface iface.
		"""
		assert self.__chain in (CHAIN_FORWARD, CHAIN_POSTROUTING, CHAIN_OUTPUT)
		self.__oIface = iface
		return self

	def protoUDP(self, invert=False):
		"""
		x.protoUDP([invert]) <==> x.proto(PROTO_UDP [, invert])
		"""
		return self.proto(PROTO_UDP, invert)

	def protoTCP(self, invert=False):
		"""
		x.protoTCP([invert]) <==> x.proto(PROTO_TCP [, invert])
		"""
		return self.proto(PROTO_TCP, invert)

	def protoICMP(self, invert=False):
		"""
		x.protoICMP([invert]) <==> x.proto(PROTO_ICMP [, invert])
		"""
		assert self.getIPversion() == IPTABLE_V4
		return self.proto(PROTO_ICMP, invert)

	def protoICMPv6(self, invert=False):
		"""
		x.protoICMPv6([invert]) <==> x.proto(PROTO_ICMPV6 [, invert])
		"""
		assert self.getIPversion() == IPTABLE_V6
		return self.proto(PROTO_ICMPV6, invert)

	def proto(self, proto, invert=False):
		"""
		x.proto(proto [, invert]) -> self

		Set the protocol proto.
		Set invert true for negation.
		"""
		self.__proto = Protocol(proto, invert)
		return self

	def source(self, addr, invert=False):
		"""
		x.source(addr [,invert]) -> self

		Set the source ip address addr.
		Set invert true for negation.
		"""
		ip = IP(addr)
		assert ip.getIPversion() == self.getIPversion()
		self.__src = ip
		self.__srcInvert = invert
		return self

	def src(self, addr, invert=False):
		"""
		x.src(addr [, invert]) -> self

		Alias for source(addr [,invert]).
		"""
		return self.source(addr, invert)

	def destination(self, addr, invert=False):
		"""
		x.destination(addr [,invert]) -> self

		Set the destination ip address addr.
		Set invert true for negation.
		"""
		ip = IP(addr)
		assert ip.getIPversion() == self.getIPversion()
		self.__dst = ip
		self.__dstInvert = invert
		return self

	def dst(self, addr, invert=False):
		"""
		x.dst(addr [, invert]) -> self

		Alias for destination(addr [,invert]).
		"""
		return self.destination(addr, invert)

	def dump(self):
		"""
		x.dump() -> str

		Return the rule in a format like iptable-save.
		"""
		result= ["-A %s" % self.__chain]
		if self.__src:
			if self.__srcInvert:
				result.append("!")
			result += ("-s", self.__src.dump())
		if self.__dst:
			if self.__dstInvert:
				result.append("!")
			result += ("-d", self.__dst.dump())
		if self.__iIface:
			result += ("-i", self.__iIface.dump())
		if self.__oIface:
			result += ("-o",self.__oIface.dump())
		if self.__proto:
			result.append(self.__proto.dump())
		result += [extension.dump() for extension in self.__extension]
		result += ("-j", self.dumpTarget())
		return " ".join(result)

class Interface:
	"""
	A metwork interface or a network interface pattern.
	"""
	def __init__(self, name):
		"""
		x.__init__(name)

		Constructor.
		Interface or interface pattern name.
		"""
		self.__name = name

	def dump(self):
		"""
		x.dump() -> str

		Return the interface or interface pattern in the iptables format.
		"""
		return str(self.__name)

class Protocol:
	"""
	A network protocol.
	"""
	def __init__(self, proto, invert=False):
		"""
		x.__init__(proto [, invert])

		Constructor.
		protocol proto
		For negation set invert True.
		"""
		self.__proto = proto
		self.__invert = invert

	def dump(self):
		"""
		x.dump() -> str

		Return the protocol in the iptables format.
		"""
		result = []
		if self.__invert:
			result.append("!")
		result += ("-p", str(self.__proto))
		return " ".join(result)

class IP:
	"""
	Is a IP address.
	"""
	def __init__(self, addr):
		"""
		x.__init__(addr)

		Constructor.
		IP address addr
		"""
		self.__addr = self._getIPObject_(addr)

	def _getIPObject_(self, addr):
		"""
		x._getIPObject_(addr) -> str
		"""
		if '/' in addr:
			return ipaddr.IPNetwork(addr)
		else:
			return ipaddr.IPAddress(addr)

	def getIPversion(self):
		"""
		x.getIPversion() -> int

		Return the IP version of the given IP address.
		"""
		if type(self.__addr) in (ipaddr.IPv4Network, ipaddr.IPv4Address):
			return IPTABLE_V4
		elif type(self.__addr) in (ipaddr.IPv6Network, ipaddr.IPv6Address):
			return IPTABLE_V6
		else:
			assert 1 == 0

	def dump(self):
		"""
		x.dump() -> str

		Return the IP address in the iptables format.
		"""
		return str(self.__addr)
