#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Hypatia: Module ISBN Range [hyphen]
#

__author__ = "Neko"
__license__ = 'LGPL http://www.gnu.org/licenses/lgpl.txt'
__version__ = '0.4.0'

# Interpretes current ISBN agency ranges
# Data obtained from https://www.isbn-international.org/
# https://www.isbn-international.org/export_rangemessage.xml

# ISBN Structure
#
# Prefix Element: 978, 979
# Registration Group Element
# Registrant Element
# Publication Element
# Check Digit

import time


class RangeNode(object):

    def __init__(self, start, end, length, prev = None, next = None):
        self._start = start
        self._end = end
        self._length = length
        self._prev = prev
        self._next = next

    def search(self, value):
        if (value < self._start):
            if (self._prev):
                return self._prev.search(value)
            else:
                return 0
        if (self._end < value):
            if (self._next):
                return self._prev.search(value)
            else:
                return 0
        return self._length

    def balance(self):
        pass


class RangeList(object):

    MINLIST = 10;

    def __init__(self, range, prev = None, next = None):
        self._range = range
        self._prev = None
        self._next = None

    def search(self, value):
        if (value < self._range[0][0]):
            if (self._prev):
                return self._prev.search(value)
            else:
                return 0
        if (self._range[-1][1] < value):
            if (self._next):
                return self._next.search(value)
            else:
                return 0
        for begin, end, length in self._range:
            if (begin <= value and value <= end):
                return length;
        return 0

    def balance(self):
        if (len(self._range) >= (RangeList.MINLIST + 2) and
            not self._prev and not self._next):
            lenl = (len(self._range) - RangeList.MINLIST) // 2
            self._prev = RangeList(self._range[:lenl])
            self._prev.balance()
            self._next = RangeList(self._range[-lenl:])
            self._next.balance()
            self._range = self._range[lenl:-lenl]


class ISBNRangeError(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
        return repr(self.value)


class ISBNRange(object):

    _serial = "10e63ac6-5e93-4525-a0e5-ee13bcbb398d"
    _sdate = "Fri, 5 Dec 2014 18:25:03 CET"
    _tdate = (2014, 12, 5, 18, 25, 3, 0, 1, -1)

    _range_grp = [
        ['9780000000000', '9785999999999', 1],
        ['9786000000000', '9786499999999', 3],
        ['9786500000000', '9786999999999', 0],
        ['9787000000000', '9787999999999', 1],
        ['9788000000000', '9789499999999', 2],
        ['9789500000000', '9789899999999', 3],
        ['9789900000000', '9789989999999', 4],
        ['9789990000000', '9789999999999', 5],
        ['9790000000000', '9790999999999', 0],
        ['9791000000000', '9791299999999', 2],
        ['9791300000000', '9799999999999', 0],
    ]

    _range_reg = [
        ['9780000000000', '9780199999999', 2],
        ['9780200000000', '9780699999999', 3],
        ['9780700000000', '9780849999999', 4],
        ['9780850000000', '9780899999999', 5],
        ['9780900000000', '9780949999999', 6],
        ['9780950000000', '9780999999999', 7],
        ['9781000000000', '9781099999999', 2],
        ['9781100000000', '9781329999999', 3],
        ['9781330000000', '9781399999999', 3],
        ['9781400000000', '9781549999999', 4],
        ['9781550000000', '9781869799999', 5],
        ['9781869800000', '9781998999999', 6],
        ['9781999000000', '9781999999999', 7],
        ['9782000000000', '9782199999999', 2],
        ['9782200000000', '9782349999999', 3],
        ['9782350000000', '9782399999999', 5],
        ['9782400000000', '9782699999999', 3],
        ['9782700000000', '9782839999999', 4],
        ['9782840000000', '9782899999999', 5],
        ['9782900000000', '9782949999999', 6],
        ['9782950000000', '9782999999999', 7],
        ['9783000000000', '9783029999999', 2],
        ['9783030000000', '9783033999999', 3],
        ['9783034000000', '9783036999999', 4],
        ['9783037000000', '9783039999999', 5],
        ['9783040000000', '9783199999999', 2],
        ['9783200000000', '9783699999999', 3],
        ['9783700000000', '9783849999999', 4],
        ['9783850000000', '9783899999999', 5],
        ['9783900000000', '9783949999999', 6],
        ['9783950000000', '9783953999999', 7],
        ['9783954000000', '9783969999999', 5],
        ['9783970000000', '9783989999999', 7],
        ['9783990000000', '9783994999999', 5],
        ['9783995000000', '9783999999999', 5],
        ['9784000000000', '9784199999999', 2],
        ['9784200000000', '9784699999999', 3],
        ['9784700000000', '9784849999999', 4],
        ['9784850000000', '9784899999999', 5],
        ['9784900000000', '9784949999999', 6],
        ['9784950000000', '9784999999999', 7],
        ['9785000000000', '9785004999999', 5],
        ['9785005000000', '9785009999999', 4],
        ['9785010000000', '9785199999999', 2],
        ['9785200000000', '9785420999999', 3],
        ['9785421000000', '9785429999999', 4],
        ['9785430000000', '9785430999999', 3],
        ['9785431000000', '9785439999999', 4],
        ['9785440000000', '9785440999999', 3],
        ['9785441000000', '9785449999999', 4],
        ['9785450000000', '9785699999999', 3],
        ['9785700000000', '9785849999999', 4],
        ['9785850000000', '9785899999999', 5],
        ['9785900000000', '9785909999999', 6],
        ['9785910000000', '9785919999999', 5],
        ['9785920000000', '9785929999999', 4],
        ['9785930000000', '9785949999999', 5],
        ['9785950000000', '9785950099999', 7],
        ['9785950100000', '9785979999999', 4],
        ['9785980000000', '9785989999999', 5],
        ['9785990000000', '9785990999999', 7],
        ['9785991000000', '9785999999999', 4],
        ['9786000000000', '9786000999999', 2],
        ['9786001000000', '9786004999999', 3],
        ['9786005000000', '9786008999999', 4],
        ['9786009000000', '9786009999999', 5],
        ['9786010000000', '9786011999999', 2],
        ['9786012000000', '9786016999999', 3],
        ['9786017000000', '9786017999999', 4],
        ['9786018000000', '9786018499999', 5],
        ['9786018500000', '9786019999999', 2],
        ['9786020000000', '9786020899999', 2],
        ['9786020900000', '9786021099999', 4],
        ['9786021100000', '9786021199999', 4],
        ['9786021200000', '9786021399999', 4],
        ['9786021400000', '9786021499999', 5],
        ['9786021500000', '9786021699999', 4],
        ['9786021700000', '9786021799999', 5],
        ['9786021800000', '9786021899999', 5],
        ['9786021900000', '9786021999999', 5],
        ['9786022000000', '9786026999999', 3],
        ['9786027000000', '9786027499999', 5],
        ['9786027500000', '9786027999999', 4],
        ['9786028000000', '9786029499999', 4],
        ['9786029500000', '9786029999999', 5],
        ['9786030000000', '9786030499999', 2],
        ['9786030500000', '9786034999999', 2],
        ['9786035000000', '9786037999999', 3],
        ['9786038000000', '9786038999999', 4],
        ['9786039000000', '9786039999999', 5],
        ['9786040000000', '9786044999999', 1],
        ['9786045000000', '9786048999999', 2],
        ['9786049000000', '9786049799999', 3],
        ['9786049800000', '9786049999999', 4],
        ['9786050000000', '9786050099999', 0],
        ['9786050100000', '9786050299999', 2],
        ['9786050300000', '9786050399999', 3],
        ['9786050400000', '9786050999999', 2],
        ['9786051000000', '9786053999999', 3],
        ['9786054000000', '9786055999999', 4],
        ['9786056000000', '9786058999999', 5],
        ['9786059000000', '9786059999999', 4],
        ['9786060000000', '9786060999999', 1],
        ['9786061000000', '9786064999999', 2],
        ['9786065000000', '9786067999999', 3],
        ['9786068000000', '9786069199999', 4],
        ['9786069200000', '9786069999999', 5],
        ['9786070000000', '9786073999999', 2],
        ['9786074000000', '9786077499999', 3],
        ['9786077500000', '9786079499999', 4],
        ['9786079500000', '9786079999999', 5],
        ['9786080000000', '9786080999999', 1],
        ['9786081000000', '9786081999999', 2],
        ['9786082000000', '9786084499999', 3],
        ['9786084500000', '9786086499999', 4],
        ['9786086500000', '9786086999999', 5],
        ['9786087000000', '9786089999999', 1],
        ['9786090000000', '9786093999999', 2],
        ['9786094000000', '9786097999999', 3],
        ['9786098000000', '9786099499999', 4],
        ['9786099500000', '9786099999999', 5],
        ['9786110000000', '9786119999999', 0],
        ['9786120000000', '9786122999999', 2],
        ['9786123000000', '9786123999999', 3],
        ['9786124000000', '9786124499999', 4],
        ['9786124500000', '9786124999999', 5],
        ['9786125000000', '9786129999999', 2],
        ['9786130000000', '9786139999999', 1],
        ['9786140000000', '9786143999999', 2],
        ['9786144000000', '9786147999999', 3],
        ['9786148000000', '9786149499999', 4],
        ['9786149500000', '9786149999999', 5],
        ['9786150000000', '9786150999999', 2],
        ['9786151000000', '9786154999999', 3],
        ['9786155000000', '9786157999999', 4],
        ['9786158000000', '9786158999999', 5],
        ['9786159000000', '9786159999999', 0],
        ['9786160000000', '9786161999999', 2],
        ['9786162000000', '9786166999999', 3],
        ['9786167000000', '9786168999999', 4],
        ['9786169000000', '9786169999999', 5],
        ['9786170000000', '9786174999999', 2],
        ['9786175000000', '9786176999999', 3],
        ['9786177000000', '9786178999999', 4],
        ['9786179000000', '9786179999999', 5],
        ['9786180000000', '9786181999999', 2],
        ['9786182000000', '9786184999999', 3],
        ['9786185000000', '9786187999999', 4],
        ['9786188000000', '9786189999999', 5],
        ['9786190000000', '9786191499999', 2],
        ['9786191500000', '9786196999999', 3],
        ['9786197000000', '9786198999999', 4],
        ['9786199000000', '9786199999999', 5],
        ['9786200000000', '9786209999999', 1],
        ['9786210000000', '9786212999999', 2],
        ['9786213000000', '9786213999999', 0],
        ['9786214000000', '9786215999999', 3],
        ['9786216000000', '9786217999999', 0],
        ['9786218000000', '9786218999999', 4],
        ['9786219000000', '9786219499999', 0],
        ['9786219500000', '9786219999999', 5],
        ['9787000000000', '9787099999999', 2],
        ['9787100000000', '9787499999999', 3],
        ['9787500000000', '9787799999999', 4],
        ['9787800000000', '9787899999999', 5],
        ['9787900000000', '9787999999999', 6],
        ['9788000000000', '9788019999999', 2],
        ['9788020000000', '9788069999999', 3],
        ['9788070000000', '9788084999999', 4],
        ['9788085000000', '9788089999999', 5],
        ['9788090000000', '9788099999999', 6],
        ['9788100000000', '9788119999999', 2],
        ['9788120000000', '9788169999999', 3],
        ['9788170000000', '9788184999999', 4],
        ['9788185000000', '9788189999999', 5],
        ['9788190000000', '9788199999999', 6],
        ['9788200000000', '9788219999999', 2],
        ['9788220000000', '9788268999999', 3],
        ['9788269000000', '9788269999999', 6],
        ['9788270000000', '9788289999999', 4],
        ['9788290000000', '9788298999999', 5],
        ['9788299000000', '9788299999999', 6],
        ['9788300000000', '9788319999999', 2],
        ['9788320000000', '9788359999999', 3],
        ['9788360000000', '9788369999999', 5],
        ['9788370000000', '9788384999999', 4],
        ['9788385000000', '9788389999999', 5],
        ['9788390000000', '9788399999999', 6],
        ['9788400000000', '9788413999999', 2],
        ['9788414000000', '9788414999999', 3],
        ['9788415000000', '9788419999999', 5],
        ['9788420000000', '9788469999999', 3],
        ['9788470000000', '9788484999999', 4],
        ['9788485000000', '9788489999999', 5],
        ['9788490000000', '9788491999999', 4],
        ['9788492000000', '9788492399999', 6],
        ['9788492400000', '9788492999999', 5],
        ['9788493000000', '9788494999999', 6],
        ['9788495000000', '9788496999999', 5],
        ['9788497000000', '9788499999999', 4],
        ['9788500000000', '9788519999999', 2],
        ['9788520000000', '9788554999999', 3],
        ['9788555000000', '9788559999999', 4],
        ['9788560000000', '9788569999999', 5],
        ['9788570000000', '9788584999999', 4],
        ['9788585000000', '9788589999999', 5],
        ['9788590000000', '9788597999999', 6],
        ['9788598000000', '9788599999999', 5],
        ['9788600000000', '9788629999999', 2],
        ['9788630000000', '9788659999999', 3],
        ['9788660000000', '9788679999999', 4],
        ['9788680000000', '9788689999999', 5],
        ['9788690000000', '9788699999999', 6],
        ['9788700000000', '9788729999999', 2],
        ['9788730000000', '9788739999999', 0],
        ['9788740000000', '9788764999999', 3],
        ['9788765000000', '9788769999999', 0],
        ['9788770000000', '9788779999999', 4],
        ['9788780000000', '9788784999999', 0],
        ['9788785000000', '9788794999999', 5],
        ['9788795000000', '9788796999999', 0],
        ['9788797000000', '9788799999999', 6],
        ['9788800000000', '9788819999999', 2],
        ['9788820000000', '9788859999999', 3],
        ['9788860000000', '9788884999999', 4],
        ['9788885000000', '9788889999999', 5],
        ['9788890000000', '9788890999999', 6],
        ['9788891000000', '9788892999999', 3],
        ['9788893000000', '9788893999999', 4],
        ['9788894000000', '9788894999999', 6],
        ['9788895000000', '9788899999999', 5],
        ['9788900000000', '9788924999999', 2],
        ['9788925000000', '9788954999999', 3],
        ['9788955000000', '9788984999999', 4],
        ['9788985000000', '9788994999999', 5],
        ['9788995000000', '9788996999999', 6],
        ['9788997000000', '9788998999999', 5],
        ['9788999000000', '9788999999999', 3],
        ['9789000000000', '9789019999999', 2],
        ['9789020000000', '9789049999999', 3],
        ['9789050000000', '9789069999999', 4],
        ['9789070000000', '9789079999999', 5],
        ['9789080000000', '9789084999999', 6],
        ['9789085000000', '9789089999999', 4],
        ['9789090000000', '9789090999999', 2],
        ['9789091000000', '9789093999999', 0],
        ['9789094000000', '9789094999999', 2],
        ['9789095000000', '9789099999999', 0],
        ['9789100000000', '9789119999999', 1],
        ['9789120000000', '9789149999999', 2],
        ['9789150000000', '9789164999999', 3],
        ['9789165000000', '9789169999999', 0],
        ['9789170000000', '9789179999999', 4],
        ['9789180000000', '9789184999999', 0],
        ['9789185000000', '9789194999999', 5],
        ['9789195000000', '9789196999999', 0],
        ['9789197000000', '9789199999999', 6],
        ['9789200000000', '9789259999999', 1],
        ['9789260000000', '9789279999999', 2],
        ['9789280000000', '9789289999999', 3],
        ['9789290000000', '9789294999999', 4],
        ['9789295000000', '9789298999999', 5],
        ['9789299000000', '9789299999999', 6],
        ['9789300000000', '9789309999999', 2],
        ['9789310000000', '9789349999999', 3],
        ['9789350000000', '9789379999999', 4],
        ['9789380000000', '9789394999999', 5],
        ['9789395000000', '9789399999999', 6],
        ['9789400000000', '9789459999999', 3],
        ['9789460000000', '9789489999999', 4],
        ['9789490000000', '9789499999999', 5],
        ['9789500000000', '9789504999999', 2],
        ['9789505000000', '9789508999999', 3],
        ['9789509000000', '9789509899999', 4],
        ['9789509900000', '9789509999999', 5],
        ['9789510000000', '9789511999999', 1],
        ['9789512000000', '9789515499999', 2],
        ['9789515500000', '9789518899999', 3],
        ['9789518900000', '9789519499999', 4],
        ['9789519500000', '9789519999999', 5],
        ['9789520000000', '9789521999999', 2],
        ['9789522000000', '9789524999999', 3],
        ['9789525000000', '9789525999999', 4],
        ['9789526000000', '9789526599999', 2],
        ['9789526600000', '9789526699999', 4],
        ['9789526700000', '9789526999999', 5],
        ['9789527000000', '9789527999999', 4],
        ['9789528000000', '9789529499999', 2],
        ['9789529500000', '9789529899999', 4],
        ['9789529900000', '9789529999999', 5],
        ['9789530000000', '9789530999999', 1],
        ['9789531000000', '9789531499999', 2],
        ['9789531500000', '9789535099999', 3],
        ['9789535100000', '9789535499999', 2],
        ['9789535500000', '9789535999999', 5],
        ['9789536000000', '9789539499999', 4],
        ['9789539500000', '9789539999999', 5],
        ['9789540000000', '9789542899999', 2],
        ['9789542900000', '9789542999999', 4],
        ['9789543000000', '9789547999999', 3],
        ['9789548000000', '9789548999999', 4],
        ['9789549000000', '9789549299999', 5],
        ['9789549300000', '9789549999999', 4],
        ['9789550000000', '9789551999999', 4],
        ['9789552000000', '9789554099999', 2],
        ['9789554100000', '9789554399999', 5],
        ['9789554400000', '9789554499999', 5],
        ['9789554500000', '9789554999999', 4],
        ['9789555000000', '9789555499999', 5],
        ['9789555500000', '9789557499999', 3],
        ['9789557500000', '9789557999999', 4],
        ['9789558000000', '9789559499999', 4],
        ['9789559500000', '9789559999999', 5],
        ['9789560000000', '9789561999999', 2],
        ['9789562000000', '9789566999999', 3],
        ['9789567000000', '9789569999999', 4],
        ['9789570000000', '9789570299999', 2],
        ['9789570300000', '9789570499999', 4],
        ['9789570500000', '9789571999999', 2],
        ['9789572000000', '9789572099999', 4],
        ['9789572100000', '9789572799999', 2],
        ['9789572800000', '9789573099999', 5],
        ['9789573100000', '9789574399999', 2],
        ['9789574400000', '9789578199999', 3],
        ['9789578200000', '9789579699999', 4],
        ['9789579700000', '9789579999999', 5],
        ['9789580000000', '9789585699999', 2],
        ['9789585700000', '9789585999999', 5],
        ['9789586000000', '9789587999999', 3],
        ['9789588000000', '9789589499999', 4],
        ['9789589500000', '9789589999999', 5],
        ['9789590000000', '9789591999999', 2],
        ['9789592000000', '9789596999999', 3],
        ['9789597000000', '9789598499999', 4],
        ['9789598500000', '9789599999999', 5],
        ['9789600000000', '9789601999999', 2],
        ['9789602000000', '9789606599999', 3],
        ['9789606600000', '9789606899999', 4],
        ['9789606900000', '9789606999999', 3],
        ['9789607000000', '9789608499999', 4],
        ['9789608500000', '9789609299999', 5],
        ['9789609300000', '9789609399999', 2],
        ['9789609400000', '9789609799999', 4],
        ['9789609800000', '9789609999999', 5],
        ['9789610000000', '9789611999999', 2],
        ['9789612000000', '9789615999999', 3],
        ['9789616000000', '9789618999999', 4],
        ['9789619000000', '9789619499999', 5],
        ['9789619500000', '9789619999999', 0],
        ['9789620000000', '9789621999999', 2],
        ['9789622000000', '9789626999999', 3],
        ['9789627000000', '9789628499999', 4],
        ['9789628500000', '9789628699999', 5],
        ['9789628700000', '9789628999999', 4],
        ['9789629000000', '9789629999999', 3],
        ['9789630000000', '9789631999999', 2],
        ['9789632000000', '9789636999999', 3],
        ['9789637000000', '9789638499999', 4],
        ['9789638500000', '9789638999999', 5],
        ['9789639000000', '9789639999999', 4],
        ['9789640000000', '9789641499999', 2],
        ['9789641500000', '9789642499999', 3],
        ['9789642500000', '9789642999999', 4],
        ['9789643000000', '9789645499999', 3],
        ['9789645500000', '9789648999999', 4],
        ['9789649000000', '9789649699999', 5],
        ['9789649700000', '9789649899999', 3],
        ['9789649900000', '9789649999999', 4],
        ['9789650000000', '9789651999999', 2],
        ['9789652000000', '9789655999999', 3],
        ['9789656000000', '9789656999999', 0],
        ['9789657000000', '9789657999999', 4],
        ['9789658000000', '9789658999999', 0],
        ['9789659000000', '9789659999999', 5],
        ['9789660000000', '9789661299999', 2],
        ['9789661300000', '9789661399999', 3],
        ['9789661400000', '9789661499999', 2],
        ['9789661500000', '9789661699999', 4],
        ['9789661700000', '9789661999999', 3],
        ['9789662000000', '9789662789999', 4],
        ['9789662790000', '9789662899999', 3],
        ['9789662900000', '9789662999999', 4],
        ['9789663000000', '9789666999999', 3],
        ['9789667000000', '9789668999999', 4],
        ['9789669000000', '9789669099999', 5],
        ['9789669100000', '9789669499999', 3],
        ['9789669500000', '9789669799999', 5],
        ['9789669800000', '9789669999999', 3],
        ['9789670000000', '9789670099999', 2],
        ['9789670100000', '9789670999999', 4],
        ['9789671000000', '9789671999999', 5],
        ['9789672000000', '9789672999999', 0],
        ['9789673000000', '9789674999999', 3],
        ['9789675000000', '9789675999999', 4],
        ['9789676000000', '9789678999999', 2],
        ['9789679000000', '9789679899999', 3],
        ['9789679900000', '9789679989999', 4],
        ['9789679990000', '9789679999999', 5],
        ['9789680100000', '9789683999999', 2],
        ['9789684000000', '9789684999999', 3],
        ['9789685000000', '9789687999999', 4],
        ['9789688000000', '9789688999999', 3],
        ['9789689000000', '9789689999999', 4],
        ['9789690000000', '9789691999999', 1],
        ['9789692000000', '9789692299999', 2],
        ['9789692300000', '9789692399999', 5],
        ['9789692400000', '9789693999999', 2],
        ['9789694000000', '9789697499999', 3],
        ['9789697500000', '9789699999999', 4],
        ['9789700100000', '9789705999999', 2],
        ['9789706000000', '9789708999999', 3],
        ['9789709000000', '9789709099999', 4],
        ['9789709100000', '9789709699999', 5],
        ['9789709700000', '9789709999999', 4],
        ['9789710000000', '9789710159999', 3],
        ['9789710160000', '9789710199999', 4],
        ['9789710200000', '9789710299999', 2],
        ['9789710300000', '9789710599999', 4],
        ['9789710600000', '9789714999999', 2],
        ['9789715000000', '9789718499999', 3],
        ['9789718500000', '9789719099999', 4],
        ['9789719100000', '9789719599999', 5],
        ['9789719600000', '9789719699999', 4],
        ['9789719700000', '9789719899999', 2],
        ['9789719900000', '9789719999999', 4],
        ['9789720000000', '9789721999999', 1],
        ['9789722000000', '9789725499999', 2],
        ['9789725500000', '9789727999999', 3],
        ['9789728000000', '9789729499999', 4],
        ['9789729500000', '9789729999999', 5],
        ['9789730000000', '9789730999999', 1],
        ['9789731000000', '9789731699999', 3],
        ['9789731700000', '9789731999999', 4],
        ['9789732000000', '9789735499999', 2],
        ['9789735500000', '9789737599999', 3],
        ['9789737600000', '9789738499999', 4],
        ['9789738500000', '9789738899999', 5],
        ['9789738900000', '9789739499999', 4],
        ['9789739500000', '9789739999999', 5],
        ['9789740000000', '9789741999999', 2],
        ['9789742000000', '9789746999999', 3],
        ['9789747000000', '9789748499999', 4],
        ['9789748500000', '9789748999999', 5],
        ['9789749000000', '9789749499999', 5],
        ['9789749500000', '9789749999999', 4],
        ['9789750000000', '9789750199999', 5],
        ['9789750200000', '9789752499999', 2],
        ['9789752500000', '9789755999999', 3],
        ['9789756000000', '9789759199999', 4],
        ['9789759200000', '9789759899999', 5],
        ['9789759900000', '9789759999999', 3],
        ['9789760000000', '9789763999999', 1],
        ['9789764000000', '9789765999999', 2],
        ['9789766000000', '9789767999999', 3],
        ['9789768000000', '9789769499999', 4],
        ['9789769500000', '9789769999999', 5],
        ['9789770000000', '9789771999999', 2],
        ['9789772000000', '9789774999999', 3],
        ['9789775000000', '9789776999999', 4],
        ['9789777000000', '9789778499999', 3],
        ['9789778500000', '9789778999999', 5],
        ['9789779000000', '9789779999999', 2],
        ['9789780000000', '9789781999999', 3],
        ['9789782000000', '9789782999999', 4],
        ['9789783000000', '9789787999999', 5],
        ['9789788000000', '9789788999999', 4],
        ['9789789000000', '9789789999999', 3],
        ['9789790000000', '9789790999999', 3],
        ['9789791000000', '9789791499999', 4],
        ['9789791500000', '9789791999999', 5],
        ['9789792000000', '9789792999999', 2],
        ['9789793000000', '9789793999999', 4],
        ['9789794000000', '9789797999999', 3],
        ['9789798000000', '9789799499999', 4],
        ['9789799500000', '9789799999999', 5],
        ['9789800000000', '9789801999999', 2],
        ['9789802000000', '9789805999999', 3],
        ['9789806000000', '9789809999999', 4],
        ['9789810000000', '9789811199999', 2],
        ['9789811200000', '9789811699999', 0],
        ['9789811700000', '9789811999999', 5],
        ['9789812000000', '9789812899999', 3],
        ['9789812900000', '9789812999999', 3],
        ['9789813000000', '9789813099999', 4],
        ['9789813100000', '9789813999999', 3],
        ['9789814000000', '9789819999999', 4],
        ['9789820000000', '9789820999999', 2],
        ['9789821000000', '9789826999999', 3],
        ['9789827000000', '9789828999999', 2],
        ['9789829000000', '9789829799999', 4],
        ['9789829800000', '9789829999999', 5],
        ['9789830000000', '9789830199999', 2],
        ['9789830200000', '9789831999999', 3],
        ['9789832000000', '9789833999999', 4],
        ['9789834000000', '9789834499999', 5],
        ['9789834500000', '9789834999999', 2],
        ['9789835000000', '9789837999999', 2],
        ['9789838000000', '9789838999999', 3],
        ['9789839000000', '9789839899999', 4],
        ['9789839900000', '9789839999999', 5],
        ['9789840000000', '9789843999999', 2],
        ['9789844000000', '9789847999999', 3],
        ['9789848000000', '9789848999999', 4],
        ['9789849000000', '9789849999999', 5],
        ['9789850000000', '9789853999999', 2],
        ['9789854000000', '9789855999999', 3],
        ['9789856000000', '9789858999999', 4],
        ['9789859000000', '9789859999999', 5],
        ['9789860000000', '9789861199999', 2],
        ['9789861200000', '9789865599999', 3],
        ['9789865600000', '9789867999999', 4],
        ['9789868000000', '9789869999999', 5],
        ['9789870000000', '9789870999999', 2],
        ['9789871000000', '9789871999999', 4],
        ['9789872000000', '9789872999999', 5],
        ['9789873000000', '9789873599999', 2],
        ['9789873600000', '9789873999999', 4],
        ['9789874000000', '9789874499999', 2],
        ['9789874500000', '9789874999999', 5],
        ['9789875000000', '9789878999999', 3],
        ['9789879000000', '9789879499999', 4],
        ['9789879500000', '9789879999999', 5],
        ['9789880000000', '9789881199999', 2],
        ['9789881200000', '9789881499999', 5],
        ['9789881500000', '9789881699999', 5],
        ['9789881700000', '9789881999999', 5],
        ['9789882000000', '9789887999999', 3],
        ['9789888000000', '9789889699999', 4],
        ['9789889700000', '9789889999999', 5],
        ['9789890000000', '9789891999999', 1],
        ['9789892000000', '9789895499999', 2],
        ['9789895500000', '9789897999999', 3],
        ['9789898000000', '9789899499999', 4],
        ['9789899500000', '9789899999999', 5],
        ['9789926000000', '9789926199999', 1],
        ['9789926200000', '9789926399999', 2],
        ['9789926400000', '9789926799999', 3],
        ['9789926800000', '9789926999999', 4],
        ['9789927000000', '9789927099999', 2],
        ['9789927100000', '9789927399999', 3],
        ['9789927400000', '9789927499999', 4],
        ['9789927500000', '9789927999999', 0],
        ['9789928000000', '9789928099999', 2],
        ['9789928100000', '9789928399999', 3],
        ['9789928400000', '9789928499999', 4],
        ['9789928500000', '9789928999999', 0],
        ['9789929000000', '9789929399999', 1],
        ['9789929400000', '9789929549999', 2],
        ['9789929550000', '9789929799999', 3],
        ['9789929800000', '9789929999999', 4],
        ['9789930000000', '9789930499999', 2],
        ['9789930500000', '9789930939999', 3],
        ['9789930940000', '9789930999999', 4],
        ['9789931000000', '9789931299999', 2],
        ['9789931300000', '9789931899999', 3],
        ['9789931900000', '9789931999999', 4],
        ['9789932000000', '9789932399999', 2],
        ['9789932400000', '9789932849999', 3],
        ['9789932850000', '9789932999999', 4],
        ['9789933000000', '9789933099999', 1],
        ['9789933100000', '9789933399999', 2],
        ['9789933400000', '9789933899999', 3],
        ['9789933900000', '9789933999999', 4],
        ['9789934000000', '9789934099999', 1],
        ['9789934100000', '9789934499999', 2],
        ['9789934500000', '9789934799999', 3],
        ['9789934800000', '9789934999999', 4],
        ['9789935000000', '9789935099999', 1],
        ['9789935100000', '9789935399999', 2],
        ['9789935400000', '9789935899999', 3],
        ['9789935900000', '9789935999999', 4],
        ['9789936000000', '9789936199999', 1],
        ['9789936200000', '9789936399999', 2],
        ['9789936400000', '9789936799999', 3],
        ['9789936800000', '9789936999999', 4],
        ['9789937000000', '9789937299999', 1],
        ['9789937300000', '9789937499999', 2],
        ['9789937500000', '9789937799999', 3],
        ['9789937800000', '9789937999999', 4],
        ['9789938000000', '9789938799999', 2],
        ['9789938800000', '9789938949999', 3],
        ['9789938950000', '9789938999999', 4],
        ['9789939000000', '9789939499999', 1],
        ['9789939500000', '9789939799999', 2],
        ['9789939800000', '9789939899999', 3],
        ['9789939900000', '9789939999999', 4],
        ['9789940000000', '9789940199999', 1],
        ['9789940200000', '9789940499999', 2],
        ['9789940500000', '9789940899999', 3],
        ['9789940900000', '9789940999999', 4],
        ['9789941000000', '9789941099999', 1],
        ['9789941100000', '9789941399999', 2],
        ['9789941400000', '9789941899999', 3],
        ['9789941900000', '9789941999999', 4],
        ['9789942000000', '9789942849999', 2],
        ['9789942850000', '9789942899999', 4],
        ['9789942900000', '9789942984999', 3],
        ['9789942985000', '9789942999999', 4],
        ['9789943000000', '9789943299999', 2],
        ['9789943300000', '9789943399999', 3],
        ['9789943400000', '9789943974999', 4],
        ['9789943975000', '9789943999999', 3],
        ['9789944000000', '9789944099999', 4],
        ['9789944100000', '9789944499999', 3],
        ['9789944500000', '9789944599999', 4],
        ['9789944600000', '9789944699999', 2],
        ['9789944700000', '9789944799999', 3],
        ['9789944800000', '9789944899999', 2],
        ['9789944900000', '9789944999999', 3],
        ['9789945000000', '9789945009999', 2],
        ['9789945010000', '9789945079999', 3],
        ['9789945080000', '9789945399999', 2],
        ['9789945400000', '9789945569999', 3],
        ['9789945570000', '9789945579999', 2],
        ['9789945580000', '9789945849999', 3],
        ['9789945850000', '9789945999999', 4],
        ['9789946000000', '9789946199999', 1],
        ['9789946200000', '9789946399999', 2],
        ['9789946400000', '9789946899999', 3],
        ['9789946900000', '9789946999999', 4],
        ['9789947000000', '9789947199999', 1],
        ['9789947200000', '9789947799999', 2],
        ['9789947800000', '9789947999999', 3],
        ['9789948000000', '9789948399999', 2],
        ['9789948400000', '9789948849999', 3],
        ['9789948850000', '9789948999999', 4],
        ['9789949000000', '9789949099999', 1],
        ['9789949100000', '9789949399999', 2],
        ['9789949400000', '9789949749999', 3],
        ['9789949750000', '9789949899999', 2],
        ['9789949900000', '9789949999999', 4],
        ['9789950000000', '9789950299999', 2],
        ['9789950300000', '9789950849999', 3],
        ['9789950850000', '9789950999999', 4],
        ['9789951000000', '9789951399999', 2],
        ['9789951400000', '9789951849999', 3],
        ['9789951850000', '9789951999999', 4],
        ['9789952000000', '9789952199999', 1],
        ['9789952200000', '9789952399999', 2],
        ['9789952400000', '9789952799999', 3],
        ['9789952800000', '9789952999999', 4],
        ['9789953000000', '9789953099999', 1],
        ['9789953100000', '9789953399999', 2],
        ['9789953400000', '9789953599999', 3],
        ['9789953600000', '9789953899999', 2],
        ['9789953900000', '9789953999999', 4],
        ['9789954000000', '9789954199999', 1],
        ['9789954200000', '9789954399999', 2],
        ['9789954400000', '9789954799999', 3],
        ['9789954800000', '9789954999999', 4],
        ['9789955000000', '9789955399999', 2],
        ['9789955400000', '9789955929999', 3],
        ['9789955930000', '9789955999999', 4],
        ['9789956000000', '9789956099999', 1],
        ['9789956100000', '9789956399999', 2],
        ['9789956400000', '9789956899999', 3],
        ['9789956900000', '9789956999999', 4],
        ['9789957000000', '9789957399999', 2],
        ['9789957400000', '9789957699999', 3],
        ['9789957700000', '9789957849999', 2],
        ['9789957850000', '9789957879999', 4],
        ['9789957880000', '9789957999999', 2],
        ['9789958000000', '9789958019999', 2],
        ['9789958020000', '9789958029999', 3],
        ['9789958030000', '9789958039999', 4],
        ['9789958040000', '9789958089999', 3],
        ['9789958090000', '9789958099999', 4],
        ['9789958100000', '9789958189999', 2],
        ['9789958190000', '9789958199999', 4],
        ['9789958200000', '9789958499999', 2],
        ['9789958500000', '9789958899999', 3],
        ['9789958900000', '9789958999999', 4],
        ['9789959000000', '9789959199999', 1],
        ['9789959200000', '9789959799999', 2],
        ['9789959800000', '9789959949999', 3],
        ['9789959950000', '9789959969999', 4],
        ['9789959970000', '9789959979999', 3],
        ['9789959980000', '9789959999999', 2],
        ['9789960000000', '9789960599999', 2],
        ['9789960600000', '9789960899999', 3],
        ['9789960900000', '9789960999999', 4],
        ['9789961000000', '9789961299999', 1],
        ['9789961300000', '9789961699999', 2],
        ['9789961700000', '9789961949999', 3],
        ['9789961950000', '9789961999999', 4],
        ['9789962000000', '9789962549999', 2],
        ['9789962550000', '9789962559999', 4],
        ['9789962560000', '9789962599999', 2],
        ['9789962600000', '9789962849999', 3],
        ['9789962850000', '9789962999999', 4],
        ['9789963000000', '9789963199999', 1],
        ['9789963200000', '9789963249999', 4],
        ['9789963250000', '9789963279999', 3],
        ['9789963280000', '9789963299999', 4],
        ['9789963300000', '9789963549999', 2],
        ['9789963550000', '9789963734999', 3],
        ['9789963735000', '9789963749999', 4],
        ['9789963750000', '9789963999999', 4],
        ['9789964000000', '9789964699999', 1],
        ['9789964700000', '9789964949999', 2],
        ['9789964950000', '9789964999999', 3],
        ['9789965000000', '9789965399999', 2],
        ['9789965400000', '9789965899999', 3],
        ['9789965900000', '9789965999999', 4],
        ['9789966000000', '9789966149999', 3],
        ['9789966150000', '9789966199999', 4],
        ['9789966200000', '9789966699999', 2],
        ['9789966700000', '9789966749999', 4],
        ['9789966750000', '9789966959999', 3],
        ['9789966960000', '9789966999999', 4],
        ['9789967000000', '9789967399999', 2],
        ['9789967400000', '9789967899999', 3],
        ['9789967900000', '9789967999999', 4],
        ['9789968000000', '9789968499999', 2],
        ['9789968500000', '9789968939999', 3],
        ['9789968940000', '9789968999999', 4],
        ['9789970000000', '9789970399999', 2],
        ['9789970400000', '9789970899999', 3],
        ['9789970900000', '9789970999999', 4],
        ['9789971000000', '9789971599999', 1],
        ['9789971600000', '9789971899999', 2],
        ['9789971900000', '9789971989999', 3],
        ['9789971990000', '9789971999999', 4],
        ['9789972000000', '9789972099999', 2],
        ['9789972100000', '9789972199999', 1],
        ['9789972200000', '9789972249999', 3],
        ['9789972250000', '9789972299999', 4],
        ['9789972300000', '9789972599999', 2],
        ['9789972600000', '9789972899999', 3],
        ['9789972900000', '9789972999999', 4],
        ['9789973000000', '9789973059999', 2],
        ['9789973060000', '9789973089999', 3],
        ['9789973090000', '9789973099999', 4],
        ['9789973100000', '9789973699999', 2],
        ['9789973700000', '9789973969999', 3],
        ['9789973970000', '9789973999999', 4],
        ['9789974000000', '9789974299999', 1],
        ['9789974300000', '9789974549999', 2],
        ['9789974550000', '9789974749999', 3],
        ['9789974750000', '9789974949999', 4],
        ['9789974950000', '9789974999999', 2],
        ['9789975000000', '9789975099999', 1],
        ['9789975100000', '9789975299999', 3],
        ['9789975300000', '9789975399999', 4],
        ['9789975400000', '9789975449999', 4],
        ['9789975450000', '9789975899999', 2],
        ['9789975900000', '9789975949999', 3],
        ['9789975950000', '9789975999999', 4],
        ['9789976000000', '9789976599999', 1],
        ['9789976600000', '9789976899999', 2],
        ['9789976900000', '9789976989999', 3],
        ['9789976990000', '9789976999999', 4],
        ['9789977000000', '9789977899999', 2],
        ['9789977900000', '9789977989999', 3],
        ['9789977990000', '9789977999999', 4],
        ['9789978000000', '9789978299999', 2],
        ['9789978300000', '9789978399999', 3],
        ['9789978400000', '9789978949999', 2],
        ['9789978950000', '9789978989999', 3],
        ['9789978990000', '9789978999999', 4],
        ['9789979000000', '9789979499999', 1],
        ['9789979500000', '9789979649999', 2],
        ['9789979650000', '9789979659999', 3],
        ['9789979660000', '9789979759999', 2],
        ['9789979760000', '9789979899999', 3],
        ['9789979900000', '9789979999999', 4],
        ['9789980000000', '9789980399999', 1],
        ['9789980400000', '9789980899999', 2],
        ['9789980900000', '9789980989999', 3],
        ['9789980990000', '9789980999999', 4],
        ['9789981000000', '9789981099999', 2],
        ['9789981100000', '9789981159999', 3],
        ['9789981160000', '9789981199999', 4],
        ['9789981200000', '9789981799999', 2],
        ['9789981800000', '9789981949999', 3],
        ['9789981950000', '9789981999999', 4],
        ['9789982000000', '9789982799999', 2],
        ['9789982800000', '9789982989999', 3],
        ['9789982990000', '9789982999999', 4],
        ['9789983000000', '9789983799999', 0],
        ['9789983800000', '9789983949999', 2],
        ['9789983950000', '9789983989999', 3],
        ['9789983990000', '9789983999999', 4],
        ['9789984000000', '9789984499999', 2],
        ['9789984500000', '9789984899999', 3],
        ['9789984900000', '9789984999999', 4],
        ['9789985000000', '9789985499999', 1],
        ['9789985500000', '9789985799999', 2],
        ['9789985800000', '9789985899999', 3],
        ['9789985900000', '9789985999999', 4],
        ['9789986000000', '9789986399999', 2],
        ['9789986400000', '9789986899999', 3],
        ['9789986900000', '9789986939999', 4],
        ['9789986940000', '9789986969999', 3],
        ['9789986970000', '9789986999999', 2],
        ['9789987000000', '9789987399999', 2],
        ['9789987400000', '9789987879999', 3],
        ['9789987880000', '9789987999999', 4],
        ['9789988000000', '9789988299999', 1],
        ['9789988300000', '9789988549999', 2],
        ['9789988550000', '9789988749999', 3],
        ['9789988750000', '9789988999999', 4],
        ['9789989000000', '9789989099999', 1],
        ['9789989100000', '9789989199999', 3],
        ['9789989200000', '9789989299999', 4],
        ['9789989300000', '9789989599999', 2],
        ['9789989600000', '9789989949999', 3],
        ['9789989950000', '9789989999999', 4],
        ['9789990100000', '9789990149999', 2],
        ['9789990150000', '9789990179999', 3],
        ['9789990180000', '9789990199999', 2],
        ['9789990200000', '9789990299999', 0],
        ['9789990300000', '9789990319999', 1],
        ['9789990320000', '9789990389999', 2],
        ['9789990390000', '9789990399999', 3],
        ['9789990400000', '9789990459999', 1],
        ['9789990460000', '9789990489999', 2],
        ['9789990490000', '9789990499999', 3],
        ['9789990500000', '9789990539999', 1],
        ['9789990540000', '9789990579999', 2],
        ['9789990580000', '9789990599999', 3],
        ['9789990600000', '9789990629999', 1],
        ['9789990630000', '9789990659999', 2],
        ['9789990660000', '9789990669999', 3],
        ['9789990670000', '9789990689999', 2],
        ['9789990690000', '9789990694999', 2],
        ['9789990695000', '9789990699999', 3],
        ['9789990800000', '9789990809999', 1],
        ['9789990810000', '9789990889999', 2],
        ['9789990890000', '9789990899999', 3],
        ['9789990900000', '9789990939999', 1],
        ['9789990940000', '9789990994999', 2],
        ['9789990995000', '9789990999999', 3],
        ['9789991000000', '9789991029999', 1],
        ['9789991030000', '9789991089999', 2],
        ['9789991090000', '9789991099999', 3],
        ['9789991100000', '9789991159999', 2],
        ['9789991160000', '9789991199999', 3],
        ['9789991200000', '9789991239999', 1],
        ['9789991240000', '9789991259999', 3],
        ['9789991260000', '9789991289999', 2],
        ['9789991290000', '9789991299999', 3],
        ['9789991300000', '9789991329999', 1],
        ['9789991330000', '9789991335999', 2],
        ['9789991336000', '9789991359999', 0],
        ['9789991360000', '9789991360499', 3],
        ['9789991360500', '9789991399999', 0],
        ['9789991400000', '9789991449999', 1],
        ['9789991450000', '9789991489999', 2],
        ['9789991490000', '9789991499999', 3],
        ['9789991500000', '9789991549999', 1],
        ['9789991550000', '9789991579999', 2],
        ['9789991580000', '9789991599999', 3],
        ['9789991600000', '9789991629999', 1],
        ['9789991630000', '9789991669999', 2],
        ['9789991670000', '9789991699999', 3],
        ['9789991700000', '9789991729999', 1],
        ['9789991730000', '9789991789999', 2],
        ['9789991790000', '9789991799999', 3],
        ['9789991800000', '9789991839999', 1],
        ['9789991840000', '9789991879999', 2],
        ['9789991880000', '9789991899999', 3],
        ['9789991900000', '9789991929999', 1],
        ['9789991930000', '9789991939999', 3],
        ['9789991940000', '9789991969999', 2],
        ['9789991970000', '9789991979999', 2],
        ['9789991980000', '9789991984999', 3],
        ['9789991985000', '9789991989999', 3],
        ['9789991990000', '9789991999999', 3],
        ['9789992000000', '9789992049999', 1],
        ['9789992050000', '9789992089999', 2],
        ['9789992090000', '9789992099999', 3],
        ['9789992100000', '9789992119999', 1],
        ['9789992120000', '9789992169999', 2],
        ['9789992170000', '9789992179999', 3],
        ['9789992180000', '9789992189999', 1],
        ['9789992190000', '9789992199999', 2],
        ['9789992200000', '9789992239999', 1],
        ['9789992240000', '9789992269999', 2],
        ['9789992270000', '9789992299999', 3],
        ['9789992300000', '9789992319999', 1],
        ['9789992320000', '9789992379999', 2],
        ['9789992380000', '9789992399999', 3],
        ['9789992400000', '9789992419999', 1],
        ['9789992420000', '9789992479999', 2],
        ['9789992480000', '9789992499999', 3],
        ['9789992500000', '9789992539999', 1],
        ['9789992540000', '9789992579999', 2],
        ['9789992580000', '9789992599999', 3],
        ['9789992600000', '9789992609999', 1],
        ['9789992610000', '9789992659999', 2],
        ['9789992660000', '9789992686999', 3],
        ['9789992687000', '9789992689999', 2],
        ['9789992690000', '9789992699999', 2],
        ['9789992700000', '9789992729999', 1],
        ['9789992730000', '9789992759999', 2],
        ['9789992760000', '9789992799999', 3],
        ['9789992800000', '9789992809999', 1],
        ['9789992810000', '9789992879999', 2],
        ['9789992880000', '9789992899999', 3],
        ['9789992900000', '9789992949999', 1],
        ['9789992950000', '9789992979999', 2],
        ['9789992980000', '9789992999999', 3],
        ['9789993000000', '9789993049999', 1],
        ['9789993050000', '9789993079999', 2],
        ['9789993080000', '9789993099999', 3],
        ['9789993100000', '9789993149999', 1],
        ['9789993150000', '9789993179999', 2],
        ['9789993180000', '9789993199999', 3],
        ['9789993200000', '9789993209999', 1],
        ['9789993210000', '9789993259999', 2],
        ['9789993260000', '9789993269999', 3],
        ['9789993270000', '9789993279999', 1],
        ['9789993280000', '9789993299999', 2],
        ['9789993300000', '9789993329999', 1],
        ['9789993330000', '9789993359999', 2],
        ['9789993360000', '9789993399999', 3],
        ['9789993400000', '9789993419999', 1],
        ['9789993420000', '9789993479999', 2],
        ['9789993480000', '9789993499999', 3],
        ['9789993500000', '9789993529999', 1],
        ['9789993530000', '9789993559999', 2],
        ['9789993560000', '9789993569999', 3],
        ['9789993570000', '9789993589999', 1],
        ['9789993590000', '9789993599999', 2],
        ['9789993600000', '9789993609999', 1],
        ['9789993610000', '9789993659999', 2],
        ['9789993660000', '9789993699999', 3],
        ['9789993700000', '9789993719999', 1],
        ['9789993720000', '9789993759999', 2],
        ['9789993760000', '9789993799999', 3],
        ['9789993800000', '9789993819999', 1],
        ['9789993820000', '9789993859999', 2],
        ['9789993860000', '9789993889999', 3],
        ['9789993890000', '9789993899999', 2],
        ['9789993900000', '9789993959999', 1],
        ['9789993960000', '9789993989999', 2],
        ['9789993990000', '9789993999999', 3],
        ['9789994000000', '9789994009999', 1],
        ['9789994010000', '9789994069999', 2],
        ['9789994070000', '9789994099999', 3],
        ['9789994100000', '9789994129999', 1],
        ['9789994130000', '9789994179999', 2],
        ['9789994180000', '9789994199999', 3],
        ['9789994200000', '9789994249999', 1],
        ['9789994250000', '9789994279999', 2],
        ['9789994280000', '9789994299999', 3],
        ['9789994300000', '9789994329999', 1],
        ['9789994330000', '9789994359999', 2],
        ['9789994360000', '9789994399999', 3],
        ['9789994400000', '9789994449999', 1],
        ['9789994450000', '9789994479999', 2],
        ['9789994480000', '9789994499999', 3],
        ['9789994500000', '9789994559999', 1],
        ['9789994560000', '9789994589999', 2],
        ['9789994590000', '9789994599999', 3],
        ['9789994600000', '9789994629999', 1],
        ['9789994630000', '9789994659999', 2],
        ['9789994660000', '9789994699999', 3],
        ['9789994700000', '9789994729999', 1],
        ['9789994730000', '9789994769999', 2],
        ['9789994770000', '9789994799999', 3],
        ['9789994800000', '9789994849999', 1],
        ['9789994850000', '9789994879999', 2],
        ['9789994880000', '9789994899999', 3],
        ['9789994900000', '9789994919999', 1],
        ['9789994920000', '9789994989999', 2],
        ['9789994990000', '9789994999999', 3],
        ['9789995000000', '9789995049999', 1],
        ['9789995050000', '9789995079999', 2],
        ['9789995080000', '9789995099999', 3],
        ['9789995100000', '9789995199999', 0],
        ['9789995200000', '9789995249999', 1],
        ['9789995250000', '9789995279999', 2],
        ['9789995280000', '9789995299999', 3],
        ['9789995300000', '9789995329999', 1],
        ['9789995330000', '9789995379999', 2],
        ['9789995380000', '9789995393999', 3],
        ['9789995394000', '9789995399999', 2],
        ['9789995400000', '9789995429999', 1],
        ['9789995430000', '9789995469999', 2],
        ['9789995470000', '9789995487999', 3],
        ['9789995488000', '9789995499999', 2],
        ['9789995500000', '9789995519999', 1],
        ['9789995520000', '9789995559999', 2],
        ['9789995560000', '9789995579999', 3],
        ['9789995580000', '9789995599999', 2],
        ['9789995600000', '9789995659999', 2],
        ['9789995660000', '9789995685999', 3],
        ['9789995686000', '9789995699999', 2],
        ['9789995700000', '9789995719999', 1],
        ['9789995720000', '9789995779999', 2],
        ['9789995780000', '9789995799999', 3],
        ['9789995800000', '9789995849999', 1],
        ['9789995850000', '9789995893999', 2],
        ['9789995894000', '9789995894999', 3],
        ['9789995895000', '9789995899999', 3],
        ['9789995900000', '9789995929999', 1],
        ['9789995930000', '9789995959999', 2],
        ['9789995960000', '9789995999999', 3],
        ['9789996000000', '9789996009999', 1],
        ['9789996010000', '9789996094999', 2],
        ['9789996095000', '9789996099999', 3],
        ['9789996100000', '9789996139999', 1],
        ['9789996140000', '9789996189999', 2],
        ['9789996190000', '9789996199999', 3],
        ['9789996200000', '9789996249999', 1],
        ['9789996250000', '9789996279999', 2],
        ['9789996280000', '9789996299999', 3],
        ['9789996300000', '9789996349999', 2],
        ['9789996350000', '9789996391999', 3],
        ['9789996392000', '9789996399999', 2],
        ['9789996400000', '9789996419999', 1],
        ['9789996420000', '9789996479999', 2],
        ['9789996480000', '9789996499999', 3],
        ['9789996500000', '9789996539999', 1],
        ['9789996540000', '9789996569999', 2],
        ['9789996570000', '9789996599999', 3],
        ['9789996600000', '9789996629999', 1],
        ['9789996630000', '9789996669999', 2],
        ['9789996670000', '9789996679999', 3],
        ['9789996680000', '9789996694999', 2],
        ['9789996695000', '9789996699999', 0],
        ['9789996700000', '9789996719999', 1],
        ['9789996720000', '9789996759999', 2],
        ['9789996760000', '9789996789999', 3],
        ['9789996790000', '9789996799999', 0],
        ['9789996800000', '9789996839999', 1],
        ['9789996840000', '9789996859999', 3],
        ['9789996860000', '9789996889999', 2],
        ['9789996890000', '9789996899999', 3],
        ['9789996900000', '9789996949999', 1],
        ['9789996950000', '9789996979999', 2],
        ['9789996980000', '9789996999999', 3],
        ['9789997000000', '9789997049999', 1],
        ['9789997050000', '9789997089999', 2],
        ['9789997090000', '9789997099999', 3],
        ['9789997100000', '9789997159999', 1],
        ['9789997160000', '9789997184999', 2],
        ['9789997185000', '9789997199999', 3],
        ['9789997200000', '9789997249999', 1],
        ['9789997250000', '9789997289999', 2],
        ['9789997290000', '9789997299999', 3],
        ['9789997300000', '9789997339999', 1],
        ['9789997340000', '9789997379999', 2],
        ['9789997380000', '9789997399999', 3],
        ['9789997400000', '9789997439999', 0],
        ['9789997440000', '9789997479999', 2],
        ['9789997480000', '9789997499999', 3],
        ['9789997500000', '9789997539999', 1],
        ['9789997540000', '9789997579999', 2],
        ['9789997580000', '9789997599999', 3],
        ['9789997600000', '9789997619999', 1],
        ['9789997620000', '9789997659999', 2],
        ['9789997660000', '9789997679999', 3],
        ['9789997680000', '9789997699999', 0],
        ['9791000000000', '9791019999999', 2],
        ['9791020000000', '9791069999999', 3],
        ['9791070000000', '9791089999999', 4],
        ['9791090000000', '9791097599999', 5],
        ['9791097600000', '9791099999999', 6],
        ['9791100000000', '9791124999999', 2],
        ['9791125000000', '9791154999999', 3],
        ['9791155000000', '9791184999999', 4],
        ['9791185000000', '9791194999999', 5],
        ['9791195000000', '9791199999999', 6],
        ['9791200000000', '9791219999999', 0],
        ['9791220000000', '9791220099999', 3],
        ['9791220100000', '9791299999999', 0],
    ]

    _tree_grp = None
    _tree_reg = None
    
    def __init__(self, url = None): # url or filename
        if (not ISBNRange._tree_grp):
            ISBNRange._tree_grp = RangeList(ISBNRange._range_grp)
            ISBNRange._tree_grp.balance()
        if (not ISBNRange._tree_reg):            
            ISBNRange._tree_reg = RangeList(ISBNRange._range_reg)
            ISBNRange._tree_reg.balance()

    @staticmethod
    def hyphensegments(isbn):
        grp = ISBNRange._tree_grp.search(isbn)
        reg = ISBNRange._tree_reg.search(isbn)
        if not grp:
            raise ISBNRangeError(isbn)
        if not reg:
            raise ISBNRangeError(isbn)
        return [3, grp, reg, 9 - grp - reg, 1]

    @staticmethod
    def hyphenformat(isbn):
        pos = []
        start = 0

        for i in ISBNRange.hyphensegments(isbn):
            pos.append(isbn[start:start + i])
            start = start + i

        return '-'.join(pos)

def _doctest ():
    import doctest
    doctest.testmod()

if __name__ == '__main__':
    _doctest()


