from jenkinsapi.jenkins import Jenkins
from .JenkinsSession import JenkinsSession
from .Config import Config
from ui.Login import Factory
import re, sys
import pickle


class MrHat:
    '''
    Interaction with the build server on MrHat to find build numbers, etc
    '''
    def __init__(self):
        session = JenkinsSession()
        counter = 0
        self.jenkins = None
        while self.jenkins is None and counter < 3:
            try:
                user = session.load_jenkins_user()
                token = session.load_jenkins_token()
                self.jenkins = Jenkins('http://mrhat.internal.radian6.com/jenkins', user, token)
                self.creds = (user, token)
            except:
                if sys.stdin.isatty() and sys.stdout.isatty():
                    login = Factory().get_login('CLI', 'Login to Jenkins...')
                else:
                    login = Factory().get_login('GUI', 'Login to Jenkins...')
                    
                login.add_prompt('user', 'Jenkins UserName', 'TEXT', session.load_jenkins_user())
                login.add_prompt('password', 'Jenkins Password', 'PASSWORD')
                    
                user = login.get_value('user')
                passwd = login.get_value('password')
                token = session.login(user, passwd)
                counter = counter + 1
            
    def get_next_maint_build(self):
        '''
        The next build for the current maint branch
        '''
        return self.get_next_build(branch='maint')
    
    def get_next_head_build(self):
        '''
        The next build for HEAD
        '''
        return self.get_next_build(branch='head')
    
    def get_next_release_build(self):
        '''
        The next build for the current release branch
        '''
        return self.get_next_build(branch='release')
    
    def get_next_build(self, branch='head'):
        '''
        Returns the next build for a specified branch
        '''
        job = self.jenkins['%s-zBuild' % branch]
        build = job.get_last_good_build()
        console = build.get_console()
        release=self.__find_value__("-DreleaseVersion=([^\s]*)", console)
        build=self.__find_value__("-DbuildVersion=([^\s]*)", console)
        b = build.split('_')
        out="MC_%s-%s" % (str(release).replace('-', '.'), self.__increment__(b[0]))
        
        return out
    
    def find_next_build(self, value):
        '''
        Returns the next build for a specified version or assumes the first build
        of an unstarted version
        :Param:value=head,release,maint or a release number
        '''
        if value in ['maint','release','head']:
            build = self.get_next_build(branch=value)
        else:
            build = self.get_next_build_for_version(value)
            
        return build
    
    def get_next_build_for_version(self, version):
        '''
        Compares a supplied version number with what is in head, release and maint
        and returns what is found, otherwise it generates a new build version
        '''
        for i in 'head', 'release', 'maint':
            out = self.get_next_build(branch=i)
            if out.count(version) > 0:
                return out
        return "MC_%s-001" % version
    
    def migrate_all_to_git_for_project(self, module):
        for job in self.get_jobs_for_cvs_repo(module):
            print("Migrate CVS job: {}".format(job))
            self.move_to_git(job)
            
        for job in self.get_jobs_for_git_module(module):
            print("Update Git job Config: {}".format(job))
            self.move_to_git(job)
    
    def get_git_repo_map(self):
        try:
            mapfile = open('git_jobmap.data', 'rb')
            jobmap = pickle.load(mapfile)
            mapfile.close()
        except:
            print("Creating Repo Map")
            jobmap = self.create_git_repo_map()
            
        return jobmap

    def create_git_repo_map(self):
        jobs = self.jenkins.keys()
        try:
            mapfile = open('git_jobmap.data', 'rb')
            jobmap = pickle.load(mapfile)
            mapfile.close()
        except:
            jobmap = {}

        for job in jobs:
            if job not in jobmap.keys():
                config = Config(self.jenkins[job].get_config())
                if config.is_configured_for_git():
                    r = config.get_git_module()
                    if r is not None:
                        jobmap[job] = r
                        mapfile = open('git_jobmap.data', 'wb')
            
                        pickle.dump(jobmap, mapfile)
                        
                        mapfile.close()     
        return jobmap
    
    def get_repo_map(self):
        try:
            mapfile = open('jobmap.data', 'rb')
            jobmap = pickle.load(mapfile)
            mapfile.close()
        except:
            print("Creating Repo Map")
            jobmap = self.create_repo_map()
            
        return jobmap
        
    def create_repo_map(self):
        jobs = self.jenkins.keys()
        try:
            mapfile = open('jobmap.data', 'rb')
            jobmap = pickle.load(mapfile)
            mapfile.close()
        except:
            jobmap = {}

        for job in jobs:
            if job not in jobmap.keys():
                config = Config(self.jenkins[job].get_config())
                r = config.get_cvs_repo()
                if r is not None:
                    jobmap[job] = r
                    mapfile = open('jobmap.data', 'wb')
        
                    pickle.dump(jobmap, mapfile)
                    
                    mapfile.close()     
        return jobmap
    
    def get_jobs_for_git_module(self, module):
        out = []
        jobmap = self.get_git_repo_map()
        for job in jobmap.keys():
            r = jobmap[job]
            if module == r:
                out.append(job)
        return out
    
    def get_jobs_for_cvs_repo(self, repo):
        out = []
        jobmap = self.get_repo_map()
        for job in jobmap.keys():
            r = jobmap[job]
            if repo == r:
                out.append(job)
        return out
    
    def remove_git_tagging(self, job):
        config = Config(self.jenkins[job].get_config())
        config.remove_git_publisher()
        out = config.remove_version_builder()
        self.jenkins[job].update_config(out)
    
    def move_to_git(self, job):
        config = Config(self.jenkins[job].get_config())
        if not config.is_configured_for_git():
            print("Migrating SCM configuration for job {} to Git".format(job))
            gitconfig = config.create_git_config()
            self.jenkins[job].update_config(gitconfig)

            if job.startswith("ci"):
                print("Adding version number builder to build wrappers")
                version = config.set_version_builder()
                self.jenkins[job].update_config(version)
         
                print("Configuring Tag Publisher")
                publish = config.set_git_publisher()
                self.jenkins[job].update_config(publish)
            else:
                print("Removing Git Tag Publisher")
                nopublish = config.remove_git_publisher()
                self.jenkins[job].update_config(nopublish)
        
            print("Ensuring Git remote name is set correctly (origin)")   
            origin = config.set_git_remote_name('origin')
            self.jenkins[job].update_config(origin)
        else:
            print("SCM is already configured for Git")
    
    def __increment__(self, build_num):
        '''
        Increments the build number and returns a new zero filled build number
        '''
        if str(build_num).startswith('SP'):
            out = 'SP'
        else:
            out = str(int(build_num) + 1).zfill(3)
            
        return out
        
    def __find_value__(self, pattern, content):
        regex = re.compile(pattern, re.MULTILINE)
        match = regex.search(content)
        out = match.group(1)
        
        return out
    
    def get_current_creds(self):
        '''
        Returns the current creds for MrHat
        '''
        return self.creds
    
