import argparse

from twisted.internet.defer import inlineCallbacks

from juju.control.utils import get_environment
from juju.machine.constraints import Constraints
from juju.state.service import ServiceStateManager


def configure_subparser(subparsers):
    sub_parser = subparsers.add_parser(
        "set-constraints",
        help=command.__doc__,
        formatter_class=argparse.RawDescriptionHelpFormatter,
        description=constraints_set.__doc__)

    sub_parser.add_argument(
        "--environment", "-e",
        help="Environment to affect")

    sub_parser.add_argument(
        "--service", "-s", default=None,
        help="Service to set constraints on")

    sub_parser.add_argument(
        "constraints",
        nargs="+",
        help="name=value for constraint to set")

    return sub_parser


def command(options):
    """Set machine constraints for the environment, or for a named service.
    """
    environment = get_environment(options)
    return constraints_set(
        environment, options.service, options.constraints)


@inlineCallbacks
def constraints_set(environment, service_name, constraint_strs):
    """
    Machine constraints allow you to pick the hardware to which your services
    will be deployed. Examples:

    $ juju set-constraints --service-name mysql mem=8G cpu=4

    $ juju set-constraints ec2-instance-type=t1.micro

    "arch", "cpu" and "mem" are always available; other constraints are
    provider-specific, and will be ignored if specified in an environment of
    the wrong kind. The recognised constraints are currently:

    * arch (CPU architecture: x86/amd64/arm; unset by default)
    * cpu (processing power in Amazon ECU; 1 by default)
    * mem (memory in [MGT]iB; 512M by default)
    * ec2-region (us-east-1 by default)
    * ec2-zone (unset by default)
    * ec2-instance-type (unset by default)
    * orchestra-classes (unset by default)
    * orchestra-name (unset by default)

    Service settings, if specified, will override environment settings, which
    will in turn override the juju defaults of mem=512M, cpu=1,
    ec2-region=us-east-1.

    New constraints set on an entity will completely replace that entity's
    pre-existing constraints.

    To override an environment constraint with the juju default when setting
    service constraints, just specify "name=" (rather than just not specifying
    the constraint at all, which will cause it to inherit the environment's
    value).
    """
    constraints = Constraints.from_strs(environment.type, constraint_strs)

    if service_name is None:
        raise NotImplementedError("Environment constraints not implemented")

    provider = environment.get_machine_provider()
    client = yield provider.connect()
    try:
        service_state_manager = ServiceStateManager(client)
        service = yield service_state_manager.get_service_state(service_name)
        yield service.set_constraints(constraints)
    finally:
        yield client.close()
