import operator
from string import ascii_lowercase
from UserDict import DictMixin

from juju.errors import ConstraintError, UnknownConstraintError


class Constraints(object, DictMixin):
    """A Constraints object encapsulates a set of machine constraints.

    Constraints objects are expected to be initially constructed using the
    `from_strs` method to parse user input; they can subsequently be serialised
    as a `data` dict and reconstructed directly from same.

    They also implement a dict interface, which exposes all constraints for
    the appropriate provider, and is the expected mode of usage for clients
    not concerned with the construction or comparison of Constraints objects.

    A Constraints object only ever contains a single "layer" of data, but can
    be combined with other Constraints objects in such a way as to produce a
    single object following the rules laid down in internals/placement-spec.

    Constraints objects can be compared, in a limited sense, by using the
    `can_satisfy` method.
    """

    def __init__(self, data):
        # To avoid inconsistency, all Constraints objects must be constructed
        # with the same set of _Constraint~s (and conflicts) in play.
        _Constraint.freeze()
        assert data.get("provider-type"), "missing provider-type"
        for k, v in data.items():
            _Constraint.get(k).convert(v)
        self._data = data

    @classmethod
    def from_strs(cls, provider, strs):
        """Create from strings (as used on the command line)"""
        data = {"provider-type": provider}
        relevant_names = _Constraint.names(provider)
        for s in strs:
            try:
                name, value = s.split("=", 1)
                constraint = _Constraint.get(name)
            except KeyError:
                raise UnknownConstraintError(name)
            except ValueError as e:
                raise ConstraintError(
                    "Could not interpret %r constraint: %s" % (s, e))
            if name not in relevant_names:
                continue
            if not constraint.visible:
                raise ConstraintError(
                    "Cannot set computed constraint: %r" % name)
            data[name] = value or constraint.default

        conflicts = set()
        for name in sorted(data):
            for conflict in sorted(_Constraint.get(name).conflicts):
                if conflict not in relevant_names:
                    continue
                if conflict in data:
                    raise ConstraintError(
                        "Ambiguous constraints: %r overlaps with %r"
                        % (name, conflict))
                conflicts.add(conflict)

        data.update(dict((conflict, None) for conflict in conflicts))
        return Constraints(data)

    def with_series(self, series):
        """Return a Constraints with the "ubuntu-series" set to `series`"""
        data = dict(self._data)
        data["ubuntu-series"] = series
        return Constraints(data)

    @property
    def complete(self):
        """Have provider-type and ubuntu-series both been set?"""
        return None not in (
            self.get("provider-type"), self.get("ubuntu-series"))

    @property
    def data(self):
        """Return a dict suitable for serialisation and reconstruction.

        Note that data contains (1) the specified value for every
        constraint that has been explicitly set, and (2) a None value for
        every constraint which conflicts with one that has been set.

        Therefore, by updating one Constraints's data with another's,
        any setting thus masked on the lower level will be preserved as None;
        consequently, Constraints~s can be collapsed onto one another without
        losing any information that is not overridden (whether implicitly or
        explicitly) by the overriding Constraints.
        """
        return dict(self._data)

    def update(self, other):
        """Overwrite `self`'s data from `other`."""
        self._data.update(other.data)

    def can_satisfy(self, other):
        """Can a machine with constraints `self` be used for a unit with
        constraints `other`? ie ::

            if machine_constraints.can_satisfy(unit_constraints):
                # place unit on machine
        """
        if not (self.complete and other.complete):
            raise ConstraintError("Cannot compare incomplete constraints")

        for (name, unit_value) in other.items():
            if unit_value is None:
                # The unit doesn't care; any machine value will be fine.
                continue
            machine_value = self[name]
            if machine_value is None:
                # The unit *does* care, and the machine value isn't specified,
                # so we can't guarantee a match. If we were to update machine
                # constraints after provisioning (ie when we knew the values of
                # the constraints left unspecified) we'd hit this branch less
                # often.
                # We may also need to do something clever here to get sensible
                # machine reuse on ec2 -- in what circumstances, if ever, is it
                # OK to place a unit specced for one instance-type on a machine
                # of another type? Does it matter if either or both were derived
                # from generic constraints? What about cost?
                return False
            if not _Constraint.get(name).can_satisfy(machine_value, unit_value):
                # The machine's value is definitely not ok for the unit.
                return False

        return True

    # DictMixin methods

    def keys(self):
        return _Constraint.names(self._data.get("provider-type"))

    def __getitem__(self, name):
        if name not in self.keys():
            raise KeyError(name)
        constraint = _Constraint.get(name)
        raw_value = self.data.get(name, constraint.default)
        return constraint.convert(raw_value)


#======================================================================
# Constraint type registration

def _dont_convert(s):
    return s


class _Constraint(object):

    _registry = {}
    _conflicts = {}
    _frozen = False

    def __init__(self, name, default, converter, comparer, provider, visible):
        self._name = name
        self._default = default
        self._converter = converter
        self._comparer = comparer
        self._provider = provider
        self._visible = visible

    @classmethod
    def freeze(cls):
        """Once called, prevents register and register_conflicts from working.

        Intent is to enforce consistency of Constraints construction and
        overriding: if the set of constraints or the registered overlaps
        were to change at runtime, the same operations could end up
        producing different results.
        """
        cls._frozen = True

    @classmethod
    def register(cls, name, default=None, converter=_dont_convert,
                 comparer=operator.eq, provider=None, visible=True):
        """Define a constraint.

        :param str name: The constraint's name
        :param default: The default value as a str, or None to indicate "unset"
        :param converter: Function to convert str value to "real" value
        :param comparer: Function used to determine whether one constraint
            satisfies another
        :param provider: The name of the provider for which this constraint is
            meaningful (None indicates always meaningful)
        :param bool visible: If False, indicates a computed constraint which
            should not be settable by a user.

        In service of consistency, is an error to attempt to register a new
        constraint once a Constraints object has been created.
        """
        assert not cls._frozen
        inst = cls(name, default, converter, comparer, provider, visible)
        cls._registry[name] = inst
        cls._conflicts[name] = set()

    @classmethod
    def register_conflicts(cls, reds, blues):
        """Set cross-constraint override behaviour.

        :param reds: list of constraint names which affect all constraints
            specified in `blues`
        :param blues: list of constraint names which affect all constraints
            specified in `reds`

        When two constraints conflict:

        * It is an error to set both constraints in the same Constraints.
        * When a Constraints overrides another which specifies a conflicting
          constraint, the value in the overridden Constraints is cleared.

        In service of consistency, is an error to attempt to register any new
        conflicts once a Constraints object has been created.
        """
        assert not cls._frozen
        for red in reds:
            cls._conflicts[red].update(blues)
        for blue in blues:
            cls._conflicts[blue].update(reds)

    @classmethod
    def get(cls, name):
        try:
            return cls._registry[name]
        except KeyError:
            raise UnknownConstraintError(name)

    @classmethod
    def names(cls, provider):
        return [
            name for (name, constraint) in cls._registry.items()
            if constraint.provider in set([None, provider])]

    @property
    def conflicts(self):
        return self._conflicts[self._name]

    @property
    def default(self):
        return self._default

    @property
    def provider(self):
        return self._provider

    @property
    def visible(self):
        return self._visible

    def convert(self, s):
        if s is None:
            return
        try:
            return self._converter(s)
        except ValueError as e:
            raise ConstraintError(
                "Bad %r constraint %r: %s" % (self._name, s, e))

    def can_satisfy(self, candidate, benchmark):
        return self._comparer(candidate, benchmark)


#======================================================================
# Generic (but not user-visible) constraints; always relevant
_Constraint.register("ubuntu-series", visible=False)
_Constraint.register("provider-type", visible=False)


#======================================================================
# Generic (user-visible) constraints; always relevant

_VALID_ARCHS = ("i386", "amd64", "arm")

_MEGABYTES = 1
_GIGABYTES = _MEGABYTES * 1024
_TERABYTES = _GIGABYTES * 1024
_MEM_SUFFIXES = {"M": _MEGABYTES, "G": _GIGABYTES, "T": _TERABYTES}


def _convert_arch(s):
    if s in _VALID_ARCHS:
        return s
    raise ValueError("unknown architecture")


def _convert_cpu(s):
    value = float(s)
    if value > 0:
        return value
    raise ValueError("must be non-negative")


def _convert_mem(s):
    if s[-1] in _MEM_SUFFIXES:
        value = float(s[:-1]) * _MEM_SUFFIXES[s[-1]]
    else:
        value = float(s)
    if value > 0:
        return value
    raise ValueError("must be non-negative")


_Constraint.register("arch", converter=_convert_arch)
_Constraint.register(
    "mem", default="512M", converter=_convert_mem, comparer=operator.ge)
_Constraint.register(
    "cpu", default="1", converter=_convert_cpu, comparer=operator.ge)


#======================================================================
# Orchestra-only

def _convert_orchestra_classes(s):
    return s.split(",")


def _compare_orchestra_classes(candidate, benchmark):
    return set(candidate) >= set(benchmark)


_Constraint.register(
    "orchestra-classes", converter=_convert_orchestra_classes,
    comparer=_compare_orchestra_classes, provider="orchestra")
_Constraint.register("orchestra-name", provider="orchestra")


#======================================================================
# EC2-only

def _convert_ec2_zone(s):
    if s not in ascii_lowercase:
        raise ValueError("expected lowercase ascii char")
    return s


_Constraint.register("ec2-instance-type", provider="ec2")
_Constraint.register("ec2-zone", converter=_convert_ec2_zone, provider="ec2")


#======================================================================
# All conflicts

_Constraint.register_conflicts(["orchestra-name"], ["orchestra-classes"])
_Constraint.register_conflicts(
    ["ec2-instance-type", "orchestra-name"], ["arch", "cpu", "mem"])

#=====================================================================
# Ensure consistency

_Constraint.freeze()
