from juju.errors import ConstraintError
from juju.lib.testing import TestCase
from juju.machine.constraints import Constraints


generic_defaults = {
    "arch": None, "cpu": 1, "mem": 512,
    "ubuntu-series": None, "provider-type": None}
dummy_defaults = dict(generic_defaults, **{
    "provider-type": "dummy"})
ec2_defaults = dict(generic_defaults, **{
    "provider-type": "ec2",
    "ec2-zone": None,
    "ec2-instance-type": None})
orchestra_defaults = dict(generic_defaults, **{
    "provider-type": "orchestra",
    "orchestra-name": None,
    "orchestra-classes": None})

all_providers = ["dummy", "ec2", "orchestra"]

class ConstraintsTestCase(TestCase):

    def assert_error(self, message, *raises_args):
        e = self.assertRaises(ConstraintError, *raises_args)
        self.assertEquals(str(e), message)

    def assert_roundtrip_equal(self, constraints, expected):
        self.assertEquals(constraints, expected)
        self.assertEquals(Constraints(constraints.data), expected)


class ConstraintsTest(ConstraintsTestCase):

    def test_defaults(self):
        constraints = Constraints.from_strs("orchestra", [])
        self.assert_roundtrip_equal(constraints, orchestra_defaults)
        constraints = Constraints.from_strs("ec2", [])
        self.assert_roundtrip_equal(constraints, ec2_defaults)
        constraints = Constraints.from_strs("dummy", [])
        self.assert_roundtrip_equal(constraints, dummy_defaults)

    def test_complete(self):
        incomplete_constraints = Constraints.from_strs("womble", [])
        self.assertFalse(incomplete_constraints.complete)
        complete_constraints = incomplete_constraints.with_series("wandering")
        self.assertTrue(complete_constraints.complete)

    def assert_invalid(self, message, providers, *constraint_strs):
        for provider in providers:
            self.assert_error(
                message, Constraints.from_strs, provider, constraint_strs)

        if len(providers) != len(all_providers):
            # Check it *is* valid for other providers
            for provider in all_providers:
                if provider in providers:
                    continue
                Constraints.from_strs(provider, constraint_strs)

    def test_invalid_input(self):
        """Reject nonsense constraints"""
        self.assert_invalid(
            "Could not interpret 'BLAH' constraint: need more than 1 value to "
            "unpack",
            all_providers, "BLAH")
        self.assert_invalid(
            "Unknown constraint: 'foo'",
            all_providers, "foo=", "bar=")

    def test_invalid_constraints(self):
        """Reject nonsensical constraint values"""
        self.assert_invalid(
            "Bad 'arch' constraint 'leg': unknown architecture", all_providers,
            "arch=leg")
        self.assert_invalid(
            "Bad 'cpu' constraint '-1': must be non-negative", all_providers,
            "cpu=-1")
        self.assert_invalid(
            "Bad 'cpu' constraint 'fish': could not convert string to float: "
            "fish",
            all_providers, "cpu=fish")
        self.assert_invalid(
            "Bad 'mem' constraint '-1': must be non-negative", all_providers,
            "mem=-1")
        self.assert_invalid(
            "Bad 'mem' constraint '4P': invalid literal for float(): 4P",
            all_providers, "mem=4P")
        self.assert_invalid(
            "Bad 'ec2-zone' constraint 'Q': expected lowercase ascii char",
            ["ec2"], "ec2-zone=Q")

    def test_hidden_constraints(self):
        """Reject attempts to explicitly specify computed constraints"""
        self.assert_invalid(
            "Cannot set computed constraint: 'ubuntu-series'", all_providers,
            "ubuntu-series=cheesy")
        self.assert_invalid(
            "Cannot set computed constraint: 'provider-type'", all_providers,
            "provider-type=dummy")

    def test_overlap_ec2(self):
        """Overlapping ec2 constraints should be detected"""
        self.assert_invalid(
            "Ambiguous constraints: 'arch' overlaps with 'ec2-instance-type'",
            ["ec2"], "ec2-instance-type=m1.small", "arch=i386")
        self.assert_invalid(
            "Ambiguous constraints: 'cpu' overlaps with 'ec2-instance-type'",
            ["ec2"], "ec2-instance-type=m1.small", "cpu=1")
        self.assert_invalid(
            "Ambiguous constraints: 'ec2-instance-type' overlaps with 'mem'",
            ["ec2"], "ec2-instance-type=m1.small", "mem=2G")

    def test_overlap_orchestra(self):
        """Overlapping orchestra constraints should be detected"""
        self.assert_invalid(
            "Ambiguous constraints: 'arch' overlaps with 'orchestra-name'",
            ["orchestra"], "orchestra-name=herbert", "arch=i386")
        self.assert_invalid(
            "Ambiguous constraints: 'cpu' overlaps with 'orchestra-name'",
            ["orchestra"], "orchestra-name=herbert", "cpu=1")
        self.assert_invalid(
            "Ambiguous constraints: 'mem' overlaps with 'orchestra-name'",
            ["orchestra"], "orchestra-name=herbert", "mem=2G")
        self.assert_invalid(
            "Ambiguous constraints: 'orchestra-classes' overlaps with "
            "'orchestra-name'",
            ["orchestra"], "orchestra-name=herbert", "orchestra-classes=x,y")


class ConstraintsUpdateTest(ConstraintsTestCase):

    def assert_constraints(self, provider, strss, expected):
        constraints = Constraints.from_strs(provider, strss[0])
        for strs in strss[1:]:
            constraints.update(Constraints.from_strs(provider, strs))
        self.assert_roundtrip_equal(constraints, expected)

    def assert_constraints_dummy(self, strss, expected):
        expected = dict(dummy_defaults, **expected)
        self.assert_constraints("dummy", strss, expected)

    def test_constraints_dummy(self):
        """Sane constraints dicts are generated for unknown environments"""
        self.assert_constraints_dummy([[]], {})
        self.assert_constraints_dummy([["arch=arm"]], {"arch": "arm"})
        self.assert_constraints_dummy([["cpu=0.1"]], {"cpu": 0.1})
        self.assert_constraints_dummy([["mem=128"]], {"mem": 128})
        self.assert_constraints_dummy(
            [["arch=amd64", "cpu=6", "mem=1.5G"]],
            {"arch": "amd64", "cpu": 6, "mem": 1536,})

    def test_overwriting_basic(self):
        """Later values shadow earlier values"""
        self.assert_constraints_dummy(
            [["cpu=4", "mem=512"], ["arch=i386", "mem=1G"]],
            {"arch": "i386", "cpu": 4, "mem": 1024})

    def test_reset(self):
        """Empty string resets to juju default"""
        self.assert_constraints_dummy(
            [["arch=arm", "cpu=4", "mem=1024"], ["arch=", "cpu=", "mem="]],
            {"arch": None, "cpu": 1, "mem": 512})
        self.assert_constraints_dummy(
            [["arch=", "cpu=", "mem="], ["arch=arm", "cpu=4", "mem=1024"]],
            {"arch": "arm", "cpu": 4, "mem": 1024})

    def assert_constraints_ec2(self, strss, expected):
        expected = dict(ec2_defaults, **expected)
        self.assert_constraints("ec2", strss, expected)

    def test_constraints_ec2(self):
        """Sane constraints dicts are generated for ec2"""
        self.assert_constraints_ec2([[]], {})
        self.assert_constraints_ec2([["arch=arm"]], {"arch": "arm"})
        self.assert_constraints_ec2([["cpu=128"]], {"cpu": 128})
        self.assert_constraints_ec2([["mem=2G"]], {"mem": 2048})
        self.assert_constraints_ec2([["ec2-zone=b"]], {"ec2-zone": "b"})
        self.assert_constraints_ec2(
            [["arch=amd64", "cpu=32", "mem=2G", "ec2-zone=b"]],
            {"arch": "amd64", "cpu": 32, "mem": 2048, "ec2-zone": "b"})

    def test_overwriting_ec2_instance_type(self):
        """ec2-instance-type interacts correctly with arch, cpu, mem"""
        self.assert_constraints_ec2(
            [["ec2-instance-type=t1.micro"]],
            {"ec2-instance-type": "t1.micro", "cpu": None, "mem": None})
        self.assert_constraints_ec2(
            [["arch=arm", "cpu=8"], ["ec2-instance-type=t1.micro"]],
            {"ec2-instance-type": "t1.micro", "cpu": None, "mem": None})
        self.assert_constraints_ec2(
            [["ec2-instance-type=t1.micro"], ["arch=arm", "cpu=8"]],
            {"ec2-instance-type": None, "arch": "arm", "cpu": 8, "mem": None})

    def assert_constraints_orchestra(self, strss, expected):
        expected = dict(orchestra_defaults, **expected)
        self.assert_constraints("orchestra", strss, expected)

    def test_constraints_orchestra(self):
        """Sane constraints dicts are generated for orchestra"""
        self.assert_constraints_orchestra([[]], {})
        self.assert_constraints_orchestra([["arch=arm"]], {"arch": "arm"})
        self.assert_constraints_orchestra([["cpu=128"]], {"cpu": 128})
        self.assert_constraints_orchestra([["mem=0.25T"]], {"mem": 262144})
        self.assert_constraints_orchestra(
            [["orchestra-classes=x,y"]], {"orchestra-classes": ["x", "y"]})
        self.assert_constraints_orchestra(
            [["arch=i386", "cpu=2", "mem=768M", "orchestra-classes=a,b"]],
            {"arch": "i386", "cpu": 2, "mem": 768,
                "orchestra-classes": ["a", "b"]})

    def test_overwriting_orchestra_name(self):
        """orchestra-name interacts correctly with arch, cpu, mem"""
        self.assert_constraints_orchestra(
            [["orchestra-name=baggins"]],
            {"orchestra-name": "baggins", "cpu": None, "mem": None})
        self.assert_constraints_orchestra(
            [["orchestra-name=baggins"], ["arch=arm"]],
            {"orchestra-name": None, "arch": "arm", "cpu": None, "mem": None})
        self.assert_constraints_orchestra(
            [["arch=arm", "cpu=2"], ["orchestra-name=baggins"]],
            {"orchestra-name": "baggins", "arch": None, "cpu": None,
                "mem": None})

    def test_overwriting_orchestra_classes(self):
        """orchestra-classes interacts correctly with orchestra-name"""
        self.assert_constraints_orchestra(
            [["orchestra-name=baggins"], ["orchestra-classes=c,d", "cpu=2"]],
            {"orchestra-classes": ["c", "d"], "cpu": 2, "mem": None})
        self.assert_constraints_orchestra(
            [["orchestra-classes=zz,top", "arch=amd64", "cpu=2"],
                ["orchestra-name=baggins"]],
            {"orchestra-name": "baggins", "orchestra-classes": None,
                "cpu": None, "mem": None})


class ConstraintsFulfilmentTest(ConstraintsTestCase):

    def completed_constraints(self, provider="provider", series="series"):
        return Constraints.from_strs(provider, []).with_series(series)

    def assert_incomparable(self, c1, c2):
        self.assert_error(
            "Cannot compare incomplete constraints", c1.can_satisfy, c2)
        self.assert_error(
            "Cannot compare incomplete constraints", c2.can_satisfy, c1)

    def assert_match(self, c1, c2, expected):
        self.assertEquals(c1.can_satisfy(c2), expected)
        self.assertEquals(c2.can_satisfy(c1), expected)

    def test_fulfil_completeness(self):
        """can_satisfy needs to be called on and with complete Constraints~s"""
        c1 = Constraints({"provider-type": "a"})
        c2 = Constraints({"provider-type": "a"})
        self.assert_incomparable(c1, c2)
        c3 = self.completed_constraints("a")
        self.assert_incomparable(c1, c3)
        c4 = self.completed_constraints("a")
        self.assert_match(c3, c4, True)

    def test_fulfil_matches(self):
        instances = (
            Constraints({"provider-type": "a", "ubuntu-series": "x"}),
            Constraints({"provider-type": "a", "ubuntu-series": "y"}),
            Constraints({"provider-type": "b", "ubuntu-series": "x"}),
            Constraints({"provider-type": "b", "ubuntu-series": "y"}))

        for i, c1 in enumerate(instances):
            self.assert_match(c1, c1, True)
            for c2 in instances[i + 1:]:
                self.assert_match(c1, c2, False)

    def assert_can_satisfy(
            self, machine_strs, unit_strs, expected, provider="provider"):
        machine = Constraints.from_strs(provider, machine_strs)
        machine = machine.with_series("shiny")
        unit = Constraints.from_strs(provider, unit_strs)
        unit = unit.with_series("shiny")
        self.assertEquals(machine.can_satisfy(unit), expected)

        if provider != "provider":
            # Check that a different provider doesn't detect any problem,
            # because it's ignoring all off-provider constraints.
            machine = Constraints.from_strs("provider", machine_strs)
            machine = machine.with_series("shiny")
            unit = Constraints.from_strs("provider", unit_strs)
            unit = unit.with_series("shiny")
            self.assertEquals(machine.can_satisfy(unit), True)


    def test_can_satisfy(self):
        self.assert_can_satisfy([], [], True)

        self.assert_can_satisfy(["arch=arm"], [], True)
        self.assert_can_satisfy([], ["arch=arm"], False)
        self.assert_can_satisfy(["arch=i386"], ["arch=arm"], False)
        self.assert_can_satisfy(["arch=arm"], ["arch=amd64"], False)
        self.assert_can_satisfy(["arch=amd64"], ["arch=amd64"], True)

        self.assert_can_satisfy(["cpu=64"], [], True)
        self.assert_can_satisfy([], ["cpu=64"], False)
        self.assert_can_satisfy(["cpu=64"], ["cpu=32"], True)
        self.assert_can_satisfy(["cpu=32"], ["cpu=64"], False)
        self.assert_can_satisfy(["cpu=64"], ["cpu=64"], True)

        self.assert_can_satisfy(["mem=8G"], [], True)
        self.assert_can_satisfy([], ["mem=8G"], False)
        self.assert_can_satisfy(["mem=8G"], ["mem=4G"], True)
        self.assert_can_satisfy(["mem=4G"], ["mem=8G"], False)
        self.assert_can_satisfy(["mem=8G"], ["mem=8G"], True)

        self.assert_can_satisfy(
            # orchestra-name clears default cpu/mem values.
            # This may be a problem.
            ["orchestra-name=henry"], [], False, "orchestra")
        self.assert_can_satisfy(
            [], ["orchestra-name=henry"], False, "orchestra")
        self.assert_can_satisfy(
            ["orchestra-name=henry"], ["orchestra-name=jane"], False,
            "orchestra")
        self.assert_can_satisfy(
            ["orchestra-name=jane"], ["orchestra-name=henry"], False,
            "orchestra")
        self.assert_can_satisfy(
            ["orchestra-name=henry"], ["orchestra-name=henry"], True,
            "orchestra")

        self.assert_can_satisfy(
            ["orchestra-classes=a,b"], [], True, "orchestra")
        self.assert_can_satisfy(
            [], ["orchestra-classes=a,b"], False, "orchestra")
        self.assert_can_satisfy(
            ["orchestra-classes=a,b"], ["orchestra-classes=a"], True,
            "orchestra")
        self.assert_can_satisfy(
            ["orchestra-classes=a"], ["orchestra-classes=a,b"], False,
            "orchestra")
        self.assert_can_satisfy(
            ["orchestra-classes=a,b"], ["orchestra-classes=a,b"], True,
            "orchestra")
        self.assert_can_satisfy(
            ["orchestra-classes=a,b"], ["orchestra-classes=a,c"], False,
            "orchestra")

        self.assert_can_satisfy(["ec2-zone=a"], [], True, "ec2")
        self.assert_can_satisfy([], ["ec2-zone=a"], False, "ec2")
        self.assert_can_satisfy(["ec2-zone=a"], ["ec2-zone=b"], False, "ec2")
        self.assert_can_satisfy(["ec2-zone=b"], ["ec2-zone=a"], False, "ec2")
        self.assert_can_satisfy(["ec2-zone=a"], ["ec2-zone=a"], True, "ec2")

        self.assert_can_satisfy(
            # ec2-instance-type clears default cpu/mem values.
            # This may be a problem.
            ["ec2-instance-type=m1.small"], [], False, "ec2")
        self.assert_can_satisfy([], ["ec2-instance-type=m1.small"], False, "ec2")
        self.assert_can_satisfy(
            ["ec2-instance-type=m1.large"], ["ec2-instance-type=m1.small"],
            False, "ec2")
        self.assert_can_satisfy(
            ["ec2-instance-type=m1.small"], ["ec2-instance-type=m1.large"],
            False, "ec2")
        self.assert_can_satisfy(
            ["ec2-instance-type=m1.small"], ["ec2-instance-type=m1.small"],
            True, "ec2")
