# Copyright 2012 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""MaaS API client for Juju"""

from base64 import b64encode
import json
from urllib import urlencode

from juju.providers.common.utils import convert_unknown_error
from juju.providers.maas.auth import MaaSOAuthConnection
from juju.providers.maas.files import encode_multipart_data


CONSUMER_SECRET = ""


class MaaSClient(MaaSOAuthConnection):

    def __init__(self, config):
        """Initialise an API client for MaaS.

        :param config: a dict of configuration values; must contain
            'maas-server', 'maas-oauth', 'admin-secret'
        """
        self.url = config["maas-server"]
        if self.url.endswith('/'):
            # Remove the trailing slash as MaaS provides resource uris
            # with it included at the start and then goes bang if you
            # give it two slashes.
            self.url = self.url[:-1]
        self.oauth_info = config["maas-oauth"]
        self.admin_secret = config["admin-secret"]
        super(MaaSClient, self).__init__(self.oauth_info)

    def _get(self, path, params):
        """Dispatch a C{GET} call to a MaaS server.

        :param uri: The MaaS path for the endpoint to call.
        :param params: A C{dict} of parameters - or sequence of 2-tuples - to
            encode into the request.
        :return: A Deferred which fires with the result of the call.
        """
        url = "%s%s?%s" % (self.url, path, urlencode(params))
        d = self.dispatch_query(url)
        d.addCallback(json.loads)
        d.addErrback(convert_unknown_error)
        return d

    def _post(self, path, params):
        """Dispatch a C{POST} call to a MaaS server.

        :param uri: The MaaS path for the endpoint to call.
        :param params: A C{dict} of parameters to encode into the request.
        :return: A Deferred which fires with the result of the call.
        """
        url = self.url + path
        body, headers = encode_multipart_data(params, {})
        d = self.dispatch_query(url, "POST", headers=headers, data=body)
        d.addCallback(json.loads)
        d.addErrback(convert_unknown_error)
        return d

    def get_nodes(self, system_ids=None):
        """Ask MaaS to return a list of all the nodes it knows about.

        :return: A Deferred whose value is the list of nodes.
        """
        params = [("op", "list")]
        if system_ids is not None:
            params.extend(("id", system_id) for system_id in system_ids)
        return self._get("/api/1.0/nodes/", params)

    def acquire_node(self):
        """Ask MaaS to assign a node to us.

        :return: A Deferred whose value is the resource URI to the node
            that was acquired.
        """
        params = {"op": "acquire"}
        return self._post("/api/1.0/nodes/", params)

    def start_node(self, resource_uri, user_data):
        """Ask MaaS to start a node.

        :param resource_uri: The MaaS URI for the node you want to start.
        :param user_data: Any blob of data to be passed to MaaS. Must be
            possible to encode as base64.
        :return: A Deferred whose value is the resource data for the node
            as returned by get_nodes().
        """
        assert isinstance(user_data, str), (
            "User data must be a byte string.")
        params = {"op": "start", "user_data": b64encode(user_data)}
        return self._post(resource_uri, params)

    def stop_node(self, resource_uri):
        """Ask maas to shut down a node.

        :param resource_uri: The MaaS URI for the node you want to  stop.
        :return: A Deferred whose value is the resource data for the node
            as returned by get_nodes().
        """
        params = {"op": "stop"}
        return self._post(resource_uri, params)

    def release_node(self, resource_uri):
        """Ask MaaS to release a node from our ownership.

        :param resource_uri: The URI in MaaS for the node you want to release.
        :return: A Deferred which fires with the resource data for the node
            just released.
        """
        params = {"op": "release"}
        return self._post(resource_uri, params)
