# Copyright 2012 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Juju provider to connect to a MaaS server."""

import logging
from twisted.internet.defer import (
    fail,
    inlineCallbacks,
    returnValue,
    succeed,
    )

from juju.errors import ProviderError
from juju.providers.common.base import MachineProviderBase
from juju.providers.maas.files import MaaSFileStorage
from juju.providers.maas.launch import MaaSLaunchMachine
from juju.providers.maas.maas import MaaSClient
from juju.providers.maas.machine import MaaSMachine


log = logging.getLogger("juju.maas")


class MachineProvider(MachineProviderBase):
    """MachineProvider for use in a MaaS environment"""

    def __init__(self, environment_name, config):
        super(MachineProvider, self).__init__(environment_name, config)
        self.maas_client = MaaSClient(config)
        self._storage = MaaSFileStorage(config)

    @property
    def provider_type(self):
        return "maas"

    def get_file_storage(self):
        """Return a WebDAV-backed FileStorage abstraction."""
        return self._storage

    def get_serialization_data(self):
        """Get provider configuration suitable for serialization.

        We're overriding the base method so that we can deal with the
        maas-oauth data in a special way because when the environment file
        is parsed it writes the data in the config object as a list of
        its token parts.
        """
        data = super(MachineProvider, self).get_serialization_data()
        data["maas-oauth"] = ":".join(self.config["maas-oauth"])
        return data

    def start_machine(self, machine_data, master=False):
        """Start a MaaS machine.

        :param dict machine_data: desired characteristics of the new machine;
            it must include a "machine-id" key, and may include a "constraints"
            key (which is currently ignored by this provider).

        :param bool master: if True, machine will initialize the juju admin
            and run a provisioning agent, in addition to running a machine
            agent.
        """
        if "machine-id" not in machine_data:
            return fail(ProviderError(
                "Cannot launch a machine without specifying a machine-id"))
        machine_id = machine_data["machine-id"]
        return MaaSLaunchMachine(self, master).run(machine_id)

    @inlineCallbacks
    def get_machines(self, instance_ids=()):
        """List machines running in the provider.

        :param list instance_ids: ids of instances you want to get. Leave empty
            to list every
            :class:`juju.providers.maas.MaaSMachine`
            owned by this provider.

        :return: a list of
            :class:`juju.providers.maas.MaaSMachine`
        :rtype: :class:`twisted.internet.defer.Deferred`

        :raises: :exc:`juju.errors.MachinesNotFound`
        """
        instances = yield self.maas_client.get_nodes(instance_ids)
        returnValue([MaaSMachine.from_dict(i) for i in instances])

    @inlineCallbacks
    def shutdown_machines(self, machines):
        """Terminate machines associated with this provider.

        :param machines: machines to shut down
        :type machines: list of
            :class:`juju.providers.maas.MaaSMachine`

        :return: list of terminated
            :class:`juju.providers.maas.MaaSMachine`
            instances
        :rtype: :class:`twisted.internet.defer.Deferred`
        """
        if not machines:
            returnValue([])

        for machine in machines:
            if not isinstance(machine, MaaSMachine):
                raise ProviderError(
                    "Can only shut down MaaSMachines; "
                    "got a %r" % type(machine))

        ids = [m.instance_id for m in machines]
        killable_machines = yield self.get_machines(ids)
        for machine in killable_machines:
            yield self.maas_client.stop_node(machine.instance_id)
            yield self.maas_client.release_node(machine.instance_id)
        returnValue(killable_machines)

    def open_port(self, machine, machine_id, port, protocol="tcp"):
        """Authorizes `port` using `protocol` for `machine`."""
        log.warn("Firewalling is not yet implemented")
        return succeed(None)

    def close_port(self, machine, machine_id, port, protocol="tcp"):
        """Revokes `port` using `protocol` for `machine`."""
        log.warn("Firewalling is not yet implemented")
        return succeed(None)

    def get_opened_ports(self, machine, machine_id):
        """Returns a set of open (port, protocol) pairs for `machine`."""
        log.warn("Firewalling is not yet implemented")
        return succeed(set())
