# Copyright 2012 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Test cases for juju.providers.maas.maas"""

from textwrap import dedent
from twisted.internet import defer
from twisted.internet.defer import inlineCallbacks

from juju.providers.maas.maas import MaaSClient
from juju.providers.maas.tests.testing import (
    CONFIG,
    FakeMaaSHTTPConnection,
    NODE_JSON,
    TestCase,
    )


class TestMaaSConnection(TestCase):

    def test_init_config(self):
        client = MaaSClient(CONFIG)
        expected = [
            CONFIG["maas-server"], CONFIG["maas-oauth"],
            CONFIG["admin-secret"]]
        actual = [client.url, client.oauth_info, client.admin_secret]
        self.assertEqual(expected, actual)

    def test_init_config_strips_trailing_url_slash(self):
        config = CONFIG
        config["maas-server"] = "http://maas.example.com/api/1.0/"
        client = MaaSClient(CONFIG)
        self.assertEqual("http://maas.example.com/api/1.0", client.url)

    def test_oauth_sign_request(self):
        client = MaaSClient(CONFIG)
        headers = {}
        client.oauth_sign_request(
            CONFIG["maas-server"], headers)
        auth = headers['Authorization']
        match_regex = (
            'OAuth realm="", oauth_nonce="[^"]+", '
            'oauth_timestamp="[^"]+", oauth_consumer_key="maas", '
            'oauth_signature_method="PLAINTEXT", '
            'oauth_version="1.0", oauth_token="DEADBEEF1234", '
            'oauth_signature="[^"]+"')
        self.assertRegexpMatches(auth, match_regex)


class TestMaaSClientWithTwisted(TestCase):

    def get_client(self):
        """Return a MaaSClient with a FakeMaaSHTTPConnection factory."""
        log = self.setup_connection(MaaSClient, FakeMaaSHTTPConnection)
        return MaaSClient(CONFIG), log

    def test_get_nodes_returns_decoded_json(self):
        client, log = self.get_client()
        d = client.get_nodes()
        self.assertIsInstance(d, defer.Deferred)
        d.addBoth(lambda value: self.assertEqual(NODE_JSON, value))
        return d

    # TODO: Add test for get_nodes with a system_id parameter.

    @inlineCallbacks
    def test_get_nodes_connects_with_oauth_credentials(self):
        client, log = self.get_client()
        yield client.get_nodes()
        [factory_call] = [
            record for record in log if record.called == "factory"]
        self.assertIn("Authorization", factory_call.result.headers)

    @inlineCallbacks
    def test_acquire_node(self):
        client, log = self.get_client()
        maas_node_data = yield client.acquire_node()
        # Test that the returned data is a dict containing the node
        # data.
        self.assertIsInstance(maas_node_data, dict)
        self.assertIn("resource_uri", maas_node_data)

    @inlineCallbacks
    def test_acquire_node_connects_with_oauth_credentials(self):
        client, log = self.get_client()
        yield client.acquire_node()
        [factory_call] = [
            record for record in log if record.called == "factory"]
        self.assertIn("Authorization", factory_call.result.headers)

    @inlineCallbacks
    def test_start_node(self):
        resource_uri = NODE_JSON[0]["resource_uri"]
        data = "This is test data."
        client, log = self.get_client()
        returned_data = yield client.start_node(resource_uri, data)
        self.assertEqual(returned_data, NODE_JSON[0])
        # Also make sure that the connection was passed the user_data in
        # the POST data.
        expected_text = dedent(
            """
            Content-Disposition: form-data; name="user_data"

            VGhpcyBpcyB0ZXN0IGRhdGEu
            """)
        expected_text = "\r\n".join(expected_text.splitlines())
        [factory_call] = [
            record for record in log if record.called == "factory"]
        self.assertIn(expected_text, factory_call.result.data)

    @inlineCallbacks
    def test_start_node_connects_with_oauth_credentials(self):
        client, log = self.get_client()
        yield client.start_node("foo", "bar")
        [factory_call] = [
            record for record in log if record.called == "factory"]
        self.assertIn("Authorization", factory_call.result.headers)

    @inlineCallbacks
    def test_stop_node(self):
        # stop_node should power down the node and return its json data.
        resource_uri = NODE_JSON[0]["resource_uri"]
        client, log = self.get_client()
        returned_data = yield client.stop_node(resource_uri)
        self.assertEqual(returned_data, NODE_JSON[0])

    @inlineCallbacks
    def test_stop_node_connects_with_oauth_credentials(self):
        client, log = self.get_client()
        yield client.stop_node("foo")
        [factory_call] = [
            record for record in log if record.called == "factory"]
        self.assertIn("Authorization", factory_call.result.headers)

    @inlineCallbacks
    def test_release_node(self):
        """C{release_node} asks MaaS to release the node back to the pool.

        The node's new state is returned.
        """
        resource_uri = NODE_JSON[0]["resource_uri"]
        client, log = self.get_client()
        returned_data = yield client.release_node(resource_uri)
        self.assertEqual(returned_data, NODE_JSON[0])
