import logging
import os
import yaml

from twisted.internet.defer import DeferredQueue, inlineCallbacks
from juju.state.hook import RelationHookContext, RelationChange

ADDED = 0
REMOVED = 1
MODIFIED = 2

CHANGE_LABELS = {
    ADDED: "joined",
    MODIFIED: "modified",
    REMOVED: "departed"}

log = logging.getLogger("hook.scheduler")


class HookScheduler(object):
    """Schedules hooks for execution in response to changes observed.

    Performs merging of redunant events, and maintains a membership
    list for hooks, guaranteeing only nodes that have previously been
    notified of joining are present in th membership.

    Internally this utilizes a change clock, which is incremented for
    every change seen. All hook operations that would result from a
    change are indexed by change clock, and the clock is placed into
    the run queue.
    """

    def __init__(self, client, executor, unit_relation, relation_name,
                 unit_name, state_path):
        self._running = False
        self._state_path = state_path

        # The thing that will actually run the hook for us
        self._executor = executor
        # For hook context construction.
        self._client = client
        self._unit_relation = unit_relation
        self._relation_name = relation_name
        self._unit_name = unit_name

        if os.path.exists(self._state_path):
            self._load_state()
        else:
            self._create_state()

    def _create_state(self):
        # Current units (as far as the next hook should know)
        self._context_members = None
        # Current units and settings versions (as far as the queue knows)
        self._member_versions = {}
        # Tracks next operation by unit
        self._unit_ops = {}
        # Tracks unit operations by clock tick
        self._clock_units = {}
        # Run queue (clock)
        self._run_queue = DeferredQueue()
        # Artifical clock sequence
        self._clock_sequence = 0

    def _load_state(self):
        with open(self._state_path) as f:
            state = yaml.load(f.read())
            if not state:
                return self._create_state()
        self._context_members = state["context_members"]
        self._member_versions = state["member_versions"]
        self._unit_ops = state["unit_ops"]
        self._clock_units = state["clock_units"]
        self._run_queue = DeferredQueue()
        self._run_queue.pending = state["clock_queue"]
        self._clock_sequence = state["clock_sequence"]

    def _save_state(self):
        state = yaml.dump({
            "context_members": self._context_members,
            "member_versions": self._member_versions,
            "unit_ops": self._unit_ops,
            "clock_units": self._clock_units,
            "clock_queue": [
                # Strip "stop" instructions: if the lifecycle stopped us,
                # then if/when the lifecycle comes up again in a stopped
                # state, it won't start us in the first place.
                c for c in self._run_queue.pending if c is not None],
            "clock_sequence": self._clock_sequence})

        temp_path = self._state_path + "~"
        with open(temp_path, "w") as f:
            f.write(state)
        os.rename(temp_path, self._state_path)

    @property
    def running(self):
        return self._running is True

    @inlineCallbacks
    def run(self):
        """Run the hook scheduler and execution."""
        assert not self._running, "Scheduler is already running"
        try:
            with open(self._state_path, "a"):
                pass
        except IOError:
            raise AssertionError("%s is not writable!" % self._state_path)

        self._running = True
        log.debug("start")

        while self._running:
            clock = yield self._run_queue.get()

            # Check for a stop now marker value.
            if clock is None:
                if not self._running:
                    break
                continue

            # Get all the units with changes in this clock tick.
            for unit_name in self._clock_units.pop(clock):

                # Get the change for the unit.
                change_clock, change_type = self._unit_ops.pop(unit_name)

                log.debug("executing hook for %s:%s",
                          unit_name, CHANGE_LABELS[change_type])

                # Execute the hook
                yield self._execute(unit_name, change_type)
            self._save_state()

    def stop(self):
        """Stop the hook execution.

        Note this does not stop the scheduling, the relation watcher
        that feeds changes to the scheduler needs to be stopped to
        achieve that effect.
        """
        log.debug("stop")
        if not self._running:
            return
        self._running = False
        # Put a marker value onto the queue to designate, stop now.
        # This is in case we're waiting on the queue, when the stop
        # occurs.
        self._run_queue.put(None)

    def cb_change_members(self, old_units, new_units):
        log.debug("members changed: old=%s, new=%s", old_units, new_units)
        scheduled = 0
        self._clock_sequence += 1

        if self._context_members is None:
            self._context_members = list(old_units)

        if set(self._member_versions) != set(old_units):
            log.debug(
                "old does not match last recorded units: %s",
                sorted(self._member_versions))

        added = set(new_units) - set(self._member_versions)
        removed = set(self._member_versions) - set(new_units)
        self._member_versions.update(dict((unit, 0) for unit in added))
        for unit in removed:
            del self._member_versions[unit]

        for unit_name in sorted(added):
            scheduled += self._queue_change(
                unit_name, ADDED, self._clock_sequence)

        for unit_name in sorted(removed):
            scheduled += self._queue_change(
                unit_name, REMOVED, self._clock_sequence)

        if scheduled:
            self._run_queue.put(self._clock_sequence)
        self._save_state()

    def cb_change_settings(self, unit_versions):
        log.debug("settings changed: %s", unit_versions)
        scheduled = 0
        self._clock_sequence += 1
        for (unit_name, version) in unit_versions:
            if version > self._member_versions.get(unit_name, 0):
                self._member_versions[unit_name] = version
                scheduled += self._queue_change(
                    unit_name, MODIFIED, self._clock_sequence)
        if scheduled:
            self._run_queue.put(self._clock_sequence)
        self._save_state()

    def get_hook_context(self, change):
        """
        Return a hook context, corresponding to the current state of the
        system.
        """
        context_members = self._context_members or ()
        context = RelationHookContext(
            self._client, self._unit_relation, change,
            sorted(context_members), unit_name=self._unit_name)
        return context

    def _queue_change(self, unit_name, change_type, clock):
        """Queue up the node change for execution.
        """
        # If its a new change for the unit store it, and return.
        if not unit_name in self._unit_ops:
            self._unit_ops[unit_name] = (clock, change_type)
            self._clock_units.setdefault(clock, []).append(unit_name)
            return True

        # Else merge/reduce with the previous operation.
        previous_clock, previous_change = self._unit_ops[unit_name]
        change_clock, change_type = self._reduce(
            (previous_clock, previous_change),
            (self._clock_sequence, change_type))

        # If they've cancelled, remove from node and clock queues
        if change_type is None:
            del self._unit_ops[unit_name]
            self._clock_units[previous_clock].remove(unit_name)
            return False

        # Update the node queue with the merged change.
        self._unit_ops[unit_name] = (change_clock, change_type)

        # If the clock has changed, remove the old entry.
        if change_clock != previous_clock:
            self._clock_units[previous_clock].remove(unit_name)

        # If the old entry has precedence, we didn't schedule anything for
        # this clock cycle.
        if change_clock != clock:
            return False

        self._clock_units.setdefault(clock, []).append(unit_name)
        return True

    def _reduce(self, previous, new):
        """Given two change operations for a node, reduce to one operation.

        We depend on zookeeper's total ordering behavior as we don't
        attempt to handle nonsensical operation sequences like
        removed followed by a modified, or modified followed by an
        add.
        """
        previous_clock, previous_change = previous
        new_clock, new_change = new

        if previous_change == REMOVED and new_change == ADDED:
            return (new_clock, MODIFIED)

        elif previous_change == ADDED and new_change == MODIFIED:
            return (previous_clock, previous_change)

        elif previous_change == ADDED and new_change == REMOVED:
            return (None, None)

        elif previous_change == MODIFIED and new_change == REMOVED:
            return (new_clock, new_change)

        elif previous_change == MODIFIED and new_change == MODIFIED:
            return (previous_clock, previous_change)

    def _execute(self, unit_name, change_type):
        """Execute a hook script for a change.
        """
        # Determine the current members as of the change.
        if change_type == ADDED:
            self._context_members.append(unit_name)
        elif change_type == REMOVED:
            self._context_members.remove(unit_name)

        # Assemble the change and hook execution context
        change = RelationChange(
            self._relation_name, CHANGE_LABELS[change_type], unit_name)
        context = self.get_hook_context(change)

        # Execute the change.
        return self._executor(context, change)
