import logging
import os
import yaml

from twisted.internet.defer import inlineCallbacks, fail, succeed, Deferred

from juju.hooks.scheduler import HookScheduler, ADDED, MODIFIED, REMOVED
from juju.state.hook import RelationChange
from juju.state.tests.test_service import ServiceStateManagerTestBase


class SomeError(Exception):
    pass


class HookSchedulerTest(ServiceStateManagerTestBase):

    @inlineCallbacks
    def setUp(self):
        yield super(HookSchedulerTest, self).setUp()
        self.client = self.get_zookeeper_client()
        self.unit_relation = self.mocker.mock()
        self.executions = []
        self.service = yield self.add_service_from_charm("wordpress")
        self.state_file = self.makeFile()
        self.executor = self.collect_executor
        self._scheduler = None
        self.log_stream = self.capture_logging(
            "hook.scheduler", level=logging.DEBUG)

    @property
    def scheduler(self):
        # Create lazily, so we can create with a state file if we want to,
        # and swap out collect_executor when helpful to do so.
        if self._scheduler is None:
            self._scheduler = HookScheduler(
                self.client, self.executor, self.unit_relation, "",
                "wordpress/0", self.state_file)
        return self._scheduler

    def collect_executor(self, context, change):
        self.executions.append((context, change))

    def write_single_unit_state(self):
        with open(self.state_file, "w") as f:
            f.write(yaml.dump({
                "context_members": ["u-1"],
                "member_versions": {"u-1": 0},
                "unit_ops": {},
                "clock_units": {},
                "clock_queue": [],
                "clock_sequence": 1}))

    # Event reduction/coalescing cases
    def test_reduce_removed_added(self):
        """ A remove event for a node followed by an add event,
        results in a modify event.
        """
        self.write_single_unit_state()
        self.scheduler.cb_change_members(["u-1"], [])
        self.scheduler.cb_change_members([], ["u-1"])
        self.scheduler.run()
        self.assertEqual(len(self.executions), 1)
        self.assertEqual(self.executions[0][1].change_type, "modified")

        output = ("members changed: old=['u-1'], new=[]",
                  "members changed: old=[], new=['u-1']",
                  "start",
                  "executing hook for u-1:modified\n")
        self.assertEqual(self.log_stream.getvalue(), "\n".join(output))

    def test_reduce_modify_remove_add(self):
        """A modify, remove, add event for a node results in a modify.
        An extra validation of the previous test.
        """
        self.write_single_unit_state()
        self.scheduler.cb_change_settings([("u-1", 1)])
        self.scheduler.cb_change_members(["u-1"], [])
        self.scheduler.cb_change_members([], ["u-1"])
        self.scheduler.run()
        self.assertEqual(len(self.executions), 1)
        self.assertEqual(self.executions[0][1].change_type, "modified")

    def test_reduce_add_modify(self):
        """An add and modify event for a node are coalesced to an add."""
        self.scheduler.cb_change_members([], ["u-1"])
        self.scheduler.cb_change_settings([("u-1", 1)])
        self.scheduler.run()
        self.assertEqual(len(self.executions), 1)
        self.assertEqual(self.executions[0][1].change_type, "joined")

    def test_reduce_add_remove(self):
        """an add followed by a removal results in a noop."""
        self.scheduler.cb_change_members([], ["u-1"])
        self.scheduler.cb_change_members(["u-1"], [])
        self.scheduler.run()
        self.assertEqual(len(self.executions), 0)

    def test_reduce_modify_remove(self):
        """Modifying and then removing a node, results in just the removal."""
        self.write_single_unit_state()
        self.scheduler.cb_change_settings([("u-1", 1)])
        self.scheduler.cb_change_members(["u-1"], [])
        self.scheduler.run()
        self.assertEqual(len(self.executions), 1)
        self.assertEqual(self.executions[0][1].change_type, "departed")

    def test_reduce_modify_modify(self):
        """Multiple modifies get coalesced to a single modify."""
        # simulate normal startup, the first notify will always be the existing
        # membership set.
        self.scheduler.cb_change_members([], ["u-1"])
        self.scheduler.run()
        self.scheduler.stop()
        self.assertEqual(len(self.executions), 1)

        # Now continue the modify/modify reduction.
        self.scheduler.cb_change_settings([("u-1", 1)])
        self.scheduler.cb_change_settings([("u-1", 2)])
        self.scheduler.cb_change_settings([("u-1", 3)])
        self.scheduler.run()

        self.assertEqual(len(self.executions), 2)
        self.assertEqual(self.executions[1][1].change_type, "modified")

    # Other stuff.
    @inlineCallbacks
    def test_start_stop(self):
        self.assertFalse(self.scheduler.running)
        d = self.scheduler.run()
        self.assertTrue(self.scheduler.running)
        # starting multiple times results in an error
        self.assertFailure(self.scheduler.run(), AssertionError)
        self.scheduler.stop()
        self.assertFalse(self.scheduler.running)
        yield d
        # stopping multiple times is not an error
        yield self.scheduler.stop()
        self.assertFalse(self.scheduler.running)

    def test_start_stop_start(self):
        """Stop values should only be honored if the scheduler is stopped.
        """
        waits = [Deferred(), succeed(True), succeed(True), succeed(True)]
        results = []

        @inlineCallbacks
        def executor(context, change):
            yield waits[len(results)]
            results.append((context, change))

        scheduler = HookScheduler(
            self.client, executor,
            self.unit_relation, "", "wordpress/0", self.state_file)

        # Start the scheduler
        scheduler.run()

        # Now queue up some changes.
        scheduler.cb_change_members([], ["u-1"])
        scheduler.cb_change_members(["u-1"], ["u-1", "u-2"])

        # Stop the scheduler
        scheduler.stop()
        self.assertFalse(scheduler.running)

        # Finish the hook execution
        waits[0].callback(True)
        d = scheduler.run()

        # More changes
        scheduler.cb_change_settings([("u-1", 1)])
        scheduler.cb_change_settings([("u-2", 1)])

        # Scheduler should still be running.
        self.assertFalse(d.called)
        self.assertEqual(len(results), 4)

    @inlineCallbacks
    def test_run_requires_writable_state(self):
        # Induce lazy creation of scheduler, then break state file
        self.scheduler
        with open(self.state_file, "w"):
            pass
        os.chmod(self.state_file, 0)
        e = yield self.assertFailure(self.scheduler.run(), AssertionError)
        self.assertEquals(str(e), "%s is not writable!" % self.state_file)

    def test_empty_state(self):
        with open(self.state_file, "w") as f:
            f.write(yaml.dump({}))

        # Induce lazy creation to verify it can still survive
        self.scheduler

    @inlineCallbacks
    def test_membership_visibility_per_change(self):
        """Hooks are executed against changes, those changes are
        associated to a temporal timestamp, however the changes
        are scheduled for execution, and the state/time of the
        world may have advanced, to present a logically consistent
        view, we try to guarantee at a minimum, that hooks will
        always see the membership of a relation as it was at the
        time of their associated change.
        """
        self.scheduler.cb_change_members([], ["u-1", "u-2"])
        self.scheduler.cb_change_members(["u-1", "u-2"], ["u-2", "u-3"])
        self.scheduler.cb_change_settings([("u-2", 1)])

        self.scheduler.run()
        self.scheduler.stop()
        # only two reduced events, u-2, u-3 add
        self.assertEqual(len(self.executions), 2)

        # Now the first execution (u-2 add) should only see members
        # from the time of its change, not the current members. However
        # since u-1 has been subsequently removed, it no longer retains
        # an entry in the membership list.
        change_members = yield self.executions[0][0].get_members()
        self.assertEqual(change_members, ["u-2"])

        self.scheduler.cb_change_settings([("u-2", 2)])
        self.scheduler.cb_change_members(["u-2", "u-3"], ["u-2"])
        self.scheduler.run()

        self.assertEqual(len(self.executions), 4)
        self.assertEqual(self.executions[2][1].change_type, "modified")
        # Verify modify events see the correct membership.
        change_members = yield self.executions[2][0].get_members()
        self.assertEqual(change_members, ["u-2", "u-3"])

    @inlineCallbacks
    def test_membership_visibility_with_change(self):
        """We express a stronger guarantee of the above, namely that
        a hook wont see any 'active' members in a membership list, that
        it hasn't previously been given a notify of before.
        """
        with open(self.state_file, "w") as f:
            f.write(yaml.dump({
                "context_members": ["u-1", "u-2"],
                "member_versions": {"u-1": 0, "u-2": 0},
                "unit_ops": {},
                "clock_units": {},
                "clock_queue": [],
                "clock_sequence": 1}))

        self.scheduler.cb_change_members(["u-1", "u-2"], ["u-2", "u-3", "u-4"])
        self.scheduler.cb_change_settings([("u-2", 1)])

        self.scheduler.run()
        self.scheduler.stop()

        # add for u-3, u-4, remove for u-1, modify for u-2
        self.assertEqual(len(self.executions), 4)

        # Verify members for each change.
        self.assertEqual(self.executions[0][1].change_type, "joined")
        members = yield self.executions[0][0].get_members()
        self.assertEqual(members, ["u-1", "u-2", "u-3"])

        self.assertEqual(self.executions[1][1].change_type, "joined")
        members = yield self.executions[1][0].get_members()
        self.assertEqual(members, ["u-1", "u-2", "u-3", "u-4"])

        self.assertEqual(self.executions[2][1].change_type, "departed")
        members = yield self.executions[2][0].get_members()
        self.assertEqual(members, ["u-2", "u-3", "u-4"])

        self.assertEqual(self.executions[3][1].change_type, "modified")
        members = yield self.executions[2][0].get_members()
        self.assertEqual(members, ["u-2", "u-3", "u-4"])

        context = yield self.scheduler.get_hook_context(
            RelationChange("", "", ""))
        self.assertEqual((yield context.get_members()),
                         ["u-2", "u-3", "u-4"])

    @inlineCallbacks
    def test_get_relation_change_empty(self):
        """Retrieving a hook context, is possible even if no
        no hooks has previously been fired (Empty membership)."""
        context = yield self.scheduler.get_hook_context(
            RelationChange("", "", ""))
        members = yield context.get_members()
        self.assertEqual(members, [])

    @inlineCallbacks
    def test_state_is_loaded(self):
        with open(self.state_file, "w") as f:
            f.write(yaml.dump({
                "context_members": ["u-1", "u-2"],
                "member_versions": {"u-1": 5, "u-2": 2, "u-3": 0},
                "unit_ops": {"u-1": (3, MODIFIED), "u-3": (4, ADDED)},
                "clock_units": {3: ["u-1"], 4: ["u-3"]},
                "clock_queue": [3, 4],
                "clock_sequence": 4}))

        self.scheduler.run()
        while len(self.executions) < 2:
            yield self.poke_zk()
        self.scheduler.stop()

        self.assertEqual(self.executions[0][1].change_type, "modified")
        members = yield self.executions[0][0].get_members()
        self.assertEqual(members, ["u-1", "u-2"])

        self.assertEqual(self.executions[1][1].change_type, "joined")
        members = yield self.executions[1][0].get_members()
        self.assertEqual(members, ["u-1", "u-2", "u-3"])

        with open(self.state_file) as f:
            state = yaml.load(f.read())
        self.assertEquals(state, {
            "context_members": ["u-1", "u-2", "u-3"],
            "member_versions": {"u-1": 5, "u-2": 2, "u-3": 0},
            "unit_ops": {},
            "clock_units": {},
            "clock_queue": [],
            "clock_sequence": 4})

    def test_state_is_stored(self):
        with open(self.state_file, "w") as f:
            f.write(yaml.dump({
                "context_members": ["u-1", "u-2"],
                "member_versions": {"u-1": 0, "u-2": 2},
                "unit_ops": {},
                "clock_units": {},
                "clock_queue": [],
                "clock_sequence": 7}))

        self.scheduler.cb_change_members(["u-1", "u-2"], ["u-2", "u-3"])
        self.scheduler.cb_change_settings([("u-2", 3)])

        # Add a stop instruction to the queue, which should *not* be saved.
        self.scheduler.stop()

        with open(self.state_file) as f:
            state = yaml.load(f.read())
        self.assertEquals(state, {
            "context_members": ["u-1", "u-2"],
            "member_versions": {"u-2": 3, "u-3": 0},
            "unit_ops": {"u-1": (8, REMOVED),
                         "u-2": (9, MODIFIED),
                         "u-3": (8, ADDED)},
            "clock_units": {8: ["u-3", "u-1"], 9: ["u-2"]},
            "clock_queue": [8, 9],
            "clock_sequence": 9})

    @inlineCallbacks
    def test_state_stored_after_tick(self):

        def execute(context, change):
            self.execute_calls += 1
            if self.execute_calls > 1:
                return fail(SomeError())
            return succeed(None)
        self.execute_calls = 0
        self.executor = execute

        with open(self.state_file, "w") as f:
            f.write(yaml.dump({
            "context_members": ["u-1", "u-2"],
            "member_versions": {"u-1": 1, "u-2": 0, "u-3": 0},
            "unit_ops": {"u-1": (3, MODIFIED), "u-3": (4, ADDED)},
            "clock_units": {3: ["u-1"], 4: ["u-3"]},
            "clock_queue": [3, 4],
            "clock_sequence": 4}))

        d = self.scheduler.run()
        while self.execute_calls < 2:
            yield self.poke_zk()
        yield self.assertFailure(d, SomeError)
        with open(self.state_file) as f:
            self.assertEquals(yaml.load(f.read()), {
                "context_members": ["u-1", "u-2"],
                "member_versions": {"u-1": 1, "u-2": 0, "u-3": 0},
                "unit_ops": {"u-3": (4, ADDED)},
                "clock_units": {4: ["u-3"]},
                "clock_queue": [4],
                "clock_sequence": 4})

    @inlineCallbacks
    def test_state_not_stored_mid_tick(self):

        def execute(context, change):
            self.execute_called = True
            return fail(SomeError())
        self.execute_called = False
        self.executor = execute

        initial_state = {
            "context_members": ["u-1", "u-2"],
            "member_versions": {"u-1": 1, "u-2": 0, "u-3": 0},
            "unit_ops": {"u-1": (3, MODIFIED), "u-3": (4, ADDED)},
            "clock_units": {3: ["u-1"], 4: ["u-3"]},
            "clock_queue": [3, 4],
            "clock_sequence": 4}
        with open(self.state_file, "w") as f:
            f.write(yaml.dump(initial_state))

        d = self.scheduler.run()
        while not self.execute_called:
            yield self.poke_zk()
        yield self.assertFailure(d, SomeError)
        with open(self.state_file) as f:
            self.assertEquals(yaml.load(f.read()), initial_state)

    def test_ignore_equal_settings_version(self):
        """
        A modified event whose version is not greater than the latest known
        version for that unit will be ignored.
        """
        self.write_single_unit_state()
        self.scheduler.cb_change_settings([("u-1", 0)])
        self.scheduler.run()
        self.assertEquals(len(self.executions), 0)

    def test_settings_version_0_on_add(self):
        """
        When a unit is added, we assume its settings version to be 0, and
        therefore modified events with version 0 will be ignored.
        """
        self.scheduler.cb_change_members([], ["u-1"])
        self.scheduler.cb_change_settings([("u-1", 0)])
        self.scheduler.run()
        self.assertEquals(len(self.executions), 1)
        self.assertEqual(self.executions[0][1].change_type, "joined")

    def test_membership_timeslip(self):
        """
        Adds and removes are calculated based on known membership state, NOT
        on old_units.
        """
        with open(self.state_file, "w") as f:
            f.write(yaml.dump({
            "context_members": ["u-1", "u-2"],
            "member_versions": {"u-1": 0, "u-2": 0},
            "unit_ops": {},
            "clock_units": {},
            "clock_queue": [],
            "clock_sequence": 4}))

        self.scheduler.cb_change_members(["u-2"], ["u-3", "u-4"])
        self.scheduler.run()

        output = (
            "members changed: old=['u-2'], new=['u-3', 'u-4']",
            "old does not match last recorded units: ['u-1', 'u-2']",
            "start",
            "executing hook for u-3:joined",
            "executing hook for u-4:joined",
            "executing hook for u-1:departed",
            "executing hook for u-2:departed\n")
        self.assertEqual(self.log_stream.getvalue(), "\n".join(output))
