# Copyright 2012 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Test cases for juju.providers.maas.provider"""

from twisted.internet.defer import inlineCallbacks

from juju.lib.mocker import ANY
from juju.providers.maas import MachineProvider
from juju.providers.maas.maas import MaaSClient
from juju.providers.maas.tests.testing import (
    CONFIG,
    FakeMaaSHTTPConnection,
    TestCase,
    )


LOG_NAME = "juju.maas"


class TestMaaSProvider(TestCase):

    def test_create(self):
        name = "mymaas"
        provider = MachineProvider(name, CONFIG)
        self.assertEqual(provider.environment_name, name)
        self.assertEqual(provider.config, CONFIG)

    def test_config_serialization(self):
        """
        The environment config should be serializable so that the zookeeper
        agent can read it back on the master node.
        """
        keys_path = self.makeFile("my-keys")

        config = {
            "admin-secret": "foo",
            "maas-server": "http://localhost:8000",
            "maas-oauth":
                ["HskWvqQmpEwNpkQLnd", "wxHZ99gBwAucKZbwUD",
                "323ybLTwTcENZsuDGNV6KaGkp99DjWcy"],
            "authorized-keys-path": keys_path}

        expected = {
            "admin-secret": "foo",
            "maas-server": "http://localhost:8000",
            "maas-oauth":
                "HskWvqQmpEwNpkQLnd:wxHZ99gBwAucKZbwUD:"
                "323ybLTwTcENZsuDGNV6KaGkp99DjWcy",
            "authorized-keys": "my-keys"}

        provider = MachineProvider("maas", config)
        serialized = provider.get_serialization_data()
        self.assertEqual(serialized, expected)

    @inlineCallbacks
    def test_open_port(self):
        log = self.capture_logging(LOG_NAME)
        yield MachineProvider("blah", CONFIG).open_port(None, None, None)
        self.assertIn(
            "Firewalling is not yet implemented", log.getvalue())

    @inlineCallbacks
    def test_close_port(self):
        log = self.capture_logging(LOG_NAME)
        yield MachineProvider("blah", CONFIG).close_port(None, None, None)
        self.assertIn(
            "Firewalling is not yet implemented", log.getvalue())

    @inlineCallbacks
    def test_get_opened_ports(self):
        log = self.capture_logging(LOG_NAME)
        ports = yield MachineProvider(
            "blah", CONFIG).get_opened_ports(None, None)
        self.assertEquals(ports, set())
        self.assertIn(
            "Firewalling is not yet implemented", log.getvalue())

    @inlineCallbacks
    def test_get_machines(self):
        self.setup_connection(MaaSClient, FakeMaaSHTTPConnection)
        provider = MachineProvider("blah", CONFIG)
        machines = yield provider.get_machines()
        self.assertNotEqual([], machines)

    @inlineCallbacks
    def test_shutdown_machines(self):
        self.setup_connection(MaaSClient, FakeMaaSHTTPConnection)
        client = MaaSClient(CONFIG)

        # Patch the client to demonstrate that the node is stopped and
        # released when shutting down.
        mocker = self.mocker
        mock_client = mocker.patch(client)
        mocker.order()
        mock_client.stop_node(ANY)
        mocker.result(None)
        mock_client.release_node(ANY)
        mocker.result(None)
        mocker.replay()

        provider = MachineProvider("blah", CONFIG)
        provider.maas_client = mock_client
        machines = yield provider.get_machines()
        machine_to_shutdown = machines[0]
        machines_terminated = (
            yield provider.shutdown_machines([machine_to_shutdown]))
        self.assertEqual(
            [machines[0].instance_id],
            [machine.instance_id for machine in machines_terminated])
