#!/usr/bin/env python
# coding: utf-8

""" Модуль с утилитами для установки пакетов ПО.
"""

import os
import shlex
import string
import json
import crypt
import random

from sys import stdout, stderr, exit
from subprocess import CalledProcessError, Popen, PIPE

def _check_output(*args, **kwargs):
    """ Реализация subprocess.check_output, переносимая между 2.6 и 2.7
    """
    
    kwargs.update(stdout=PIPE)
    
    try:
        p = Popen(*args, **kwargs)
    except Exception, e:
        raise CalledProcessError(-1, args[0]) #, "Popen error: {0}".format(str(e)))
    
    p.wait()
    
    try:
        out = p.communicate()[0]
    except Exception, e:
        raise CalledProcessError(p.returncode, args[0]) #, "Communicate error: {0}".format(str(e)))
    
    if p.returncode != 0:
        raise CalledProcessError(p.returncode, args[0]) #, out)
    
    return out

try:
    from subprocess import check_output
except:
    check_output = _check_output        

class CommandError(Exception):
    """ Исключение, выбрасываемое при """

    def __init__(self, status, cmd, output):
        self.status = status
        self.cmd = cmd
        self.output = output


def shell_run(command_list, env={}, cwd=None):
    """ Выполняет команду через командную оболочку.
    """
    
    if not isinstance(command_list, list):
        command_list = [command_list]
    
    env.update(os.environ)
    
    for cmd in command_list:
        for line in cmd.splitlines(False):
            try:
                check_output(line, shell=True)
            except CalledProcessError, e:
                raise CommandError(e.returncode, line, None) #, e.output)

def run(command_list):
    """ Выполняет команду или последовательность команд. 
        В случае возникновения ошибки выполнение дальнейших команд прекращается и 
        выбрасывается исключение с полями:
            status - Статус последней выполненной команды.
            cmd - Текст последней выполненной команды.
            output - Вывод stderr последней выполненной команды.
    """
    
    if not isinstance(command_list, list):
        command_list = [command_list]
            
    for cmd in command_list:
        try:
            check_output(shlex.split(cmd))
        except CalledProcessError, e:
            raise CommandError(e.returncode, cmd, None) #, e.output)

def check_root():
    """ Проверка наличия доступа на уровне суперпользователя (root).
    """
    return (os.geteuid() == 0)
       
def add_apt_repos(repos):
    """ Устанавливает в системе дополнительные APT-репозитории.
        Для этого создает файл с именем /etc/apt/sources.list.d/ivideon-setup-tool.list.
        
        Репозитории устанавливаются все разом, чтобы не вызывать несколько раз apt-get update.
    """
    
    print "Adding APT repositories..."
    stdout.flush()

    need_update = False   
    
    for repo in repos:
        print '\tRepo: {0} ...'.format(repo),
        try:
            shell_run('grep /etc/apt/sources.list* -rne "{0}"'.format(repo['repo']))
            print 'Already listed.'
        except:
            need_update = True
            with open('/etc/apt/sources.list.d/ivideon-setup-tool.list', 'w') as f:
                print >>f, repo['repo']
            if 'key' in repo:
                run('apt-key adv --keyserver keyserver.ubuntu.com --recv {0}'.format(repo['key']))                
            print 'Done.'
    
    if need_update:
        print "Updating APT...",
        stdout.flush()
        run('apt-get update')    
        print "Done."
                
def install_apt_package(name, version=None):
    """ Установка пакета из APT-репозитория. Если нужный репозиторий
        ещё не подключен, то он добавляется.
    """
    
    print 'Installing APT: {0} ...'.format(name),
    stdout.flush()
    
    try:
        shell_run('dpkg -s {0} 2> /dev/null'.format(name))
    except CommandError:
        try:
            run('apt-get install --yes {0}'.format(name))
            print 'Done.'
        except CommandError, e:
            print 'FAILED.'
    else:
        print 'Already installed.'
    
def install_pip_package(name, version=None):
    """ Установка пакета из PythonPIP-репозитория.
    
        Если пакет уже установлен, то повторная установка 
        производится только при несовпадении версий.
    """
    
    print 'Installing PIP package: {0} ...'.format(name),
    stdout.flush()
    
    cmd = 'pip freeze | grep {0}'.format(name)
    if version:
        ''.join([cmd, '=={0}'.format(version)])
    
    try:        
        shell_run(cmd)
        print 'Already installed.'
    except CommandError:
        run('pip install --upgrade {0}'.format(name))    
        print 'Done.'
    
def add_user(name, password, home=None, sudoer=False):
    """ Добавление пользователя в систему.
    """
    
    print 'Adding user {0} ...'.format(name),
    stdout.flush()
    
    if sudoer:
        group = '--gid admin'
    else:
        group = ''
    
    try:
        shell_run('egrep "^{0}:" /etc/passwd'.format(name))
    except CommandError, e:
        pass       
    else:
        print "Already exists (done)."        
        return
    
    chars = string.letters + string.digits
    epassword = crypt.crypt(password, ''.join(random.sample(chars, 2)))

    try:
        shell_run('useradd --create-home --password {0} --home {1} {2} {3}'.format( epassword, home, group, name))
    except CommandError, e:
        print >>stderr, "Failed."
        print >>stderr, e.output
        raise
    print "Done."
        
def copy_file(src, dest):
    """ Копирование файлов. Выполняет рекурсивное копирование.
    """
    
    print "Copying {0} -> {1}".format(src, dest)
    
    try:
        run("cp -r {0} {1}".format(src, dest))
    except CommandError, e:
        print >>stderr, "FAILED"
        print >>stderr, "Output:", e.output
        raise
       
def file_contains(file, text):
    """ Проверка, содержит ли файл file текст text.
        Функция производит поиск всего блока text,
        не проводя удалений лишних пробелов или какой-либо ещё обработки.
    """
    f = open(file, "r")
    ftext = f.read()
    f.close()
    return ftext.find(text) >= 0

def append_to_file(file, text, unique=False):
    """ Добавление текста text к файлу file.
        Если unique == True, то добавление произойдёт только если файл ещё не содержит text.
        Возвращает True если добавление было произведено
    """
    if unique == True and file_contains(file, text):
        return False

    f = open(file, "a")
    f.write( ''.join( ["\n", text, "\n"] ) )
    f.close()
    return True
    
def make_dirs(path_list, owner=None, mode=0777):
    """ Создание каталога (рекурсивное).
        Автоматически выставляет права доступа и владельца каталога.
    """
    
    if not isinstance(path_list, list):
        path_list = [path_list]
    
    for path in path_list:
        print 'Creating directory "{0}" ...'.format(path),
        stdout.flush()
        
        try:
            os.makedirs(path, mode)
        except OSError, e:
            print "Already exists."

        if owner:
            print 'Changing owner to {0} ...'.format(owner),
            stdout.flush()
            run('chown -R {0} {1}'.format(owner, path))
            print "Done."
            
def download_bitbucket(user, password, fork, repo, rev):
    """ Скачивает архив (tar.bz) с исходниками с bitbucket.org из указанного 
        репозитория @a repo по состоянию на ревизию @a rev.
        
        @note Из-за проблем с кэшированием загрузок, выполняемых по имени 
              ветки/тэга на bitbucket'e приходится определять хэш-тэг 
              последнего changeset'a в ветке через BitBucket API и качать 
              уже его. Возможно, этот метод будет пока не слишком стабилен.
    """
    
    print 'Downloading from BitBucket: {fork}/{repo}/{tag} ...'.format(fork=fork, repo=repo, tag=rev),
    stdout.flush()
    
    api_req = "curl --stderr /dev/null -u {user}:{passwd} https://api.bitbucket.org/1.0/repositories/{fork}/{repo}/changesets?start={tag}"\
                  .format(user=user, passwd=password, repo=repo, tag=rev, fork=fork)
   
    try:
        response = check_output(api_req, shell=True)
    except Exception, e:
        print 'FAILED (API).'
        print 'Exception:', e
        return
        
    rev_data = json.loads(response)
    changeset_id = rev_data['changesets'][-1]['node']
    
    print '({0}) ...'.format(changeset_id)
    
    try:
        shell_run("""curl --stderr - -o download.tar.gz --digest -u {user}:{passwd} https://bitbucket.org/{fork}/{repo}/get/{changeset}.tar.gz
                     rm -rf {repo}
                     tar -xf download.tar.gz
                     mv {fork}-{repo}-{changeset} {repo}
                     rm download.tar.gz
                  """.format(user=user, passwd=password, repo=repo, changeset=changeset_id, fork=fork))
    except CommandError, e:
        print 'FAILED.'
    
    print 'Done.'
    
def mercurial_clone(repo, rev):
    """ Клонирование Hg-репозитория @a uri в текущую директорию.
        Репозиторий обновляется до ревизии @a rev.
        Авторизация по паролю не поддерживается; репозиторий должен
        быть либо доступен без авторизации (локально/по HTTP), либо
        по SSH с ключом.
    """
    print 'Cloning Hg: {repo} ... '.format(repo=repo),
    stdout.flush()
    
    if rev == 'working_dir':
        # Специальный случай - забирает рабочую директорию
        try:
            shell_run('cp -r {repo} .'.format(repo=repo))
        except Exception, e:
            print 'FAILED.'
            print >> stderr, 'Exception:', e
    else:
        # Реальное клонирование
        try:
            shell_run('hg clone --updaterev {rev} {repo}'.format(repo=repo, rev=rev))
        except Exception, e:
            print 'FAILED.'
            print >> stderr, 'Exception:', e
                
    print 'Done.'


def http_download(url, user=None, password=None):
    """ Загрузка файла по HTTP. Если указаны имя пользователя
        и пароль, то проходит авторизацию.
    """
    print 'Downloading HTTP: {url} ... '.format(url=url),
    stdout.flush()
    
    if user and password:
        auth_str = '--user={user} --password={password}'.format(user=user, password=password)
    else:
        auth_str = ''
    
    shell_run('wget {url} 2> /dev/null'.format(url=url))
    print 'Done.'


class XMLConfigurator(object):
    """ Класс для выполнения конфигурации, описанной в формате XML.
    """
    
    def __init__(self):
        self.env = {}
        self.os_env = os.environ.copy()
           
    def run_xml(self, file_name='setup.xml'):
        """ Выполнение XML-файла.
        """
        
        from xml.etree import ElementTree
        
        tree = ElementTree.parse(file_name)
        
        try:
            self._get_args(tree)
        except Exception, e:
            print >>stderr, "FAILED"
            print >>stderr, "Could not setup environment"
            print >>stderr, e
            exit(1)
        
        try:
            for e in tree.getroot():
                if e.tag == 'Packages':
                    self._install_packages(e)
                elif e.tag == 'User':
                    self._add_user(e)
                elif e.tag == 'File':
                    self._deploy_file(e)
                elif e.tag == 'Cmd':
                    self._run_command(e)
                elif e.tag == 'MakeDir':
                    self._make_dirs(e)
                elif e.tag == 'BitBucket':
                    self._download_bitbucket(e)
                elif e.tag == 'Hg':
                    self._clone_mercurial(e)
                elif e.tag == 'Append':
                    self._append(e)
                elif e.tag == 'Http':
                    self._http_download(e)
        except CommandError, e:
            print >>stderr, "FAILED"
            print >>stderr, "Last command:", e.cmd
            print >>stderr, "Output:", e.output
    
    def _get_args(self, e):
        """ Получение аргументов, необходимых для выполнения установки.
            Если аргументы отсутствуют в окружении (environment), то
            они запрашиваются в консоли.
        """
        
        from getpass import getpass
        
        for arg in e.findall('./Args/Arg'):
            name = arg.text
            if name not in os.environ:
                default = arg.get('default', '')
                
                if arg.get('type') == 'password':
                    read_fn = getpass
                    prompt = '*' * len(default)
                else:
                    read_fn = raw_input
                    prompt = default
                
                value = read_fn('{0} [{1}]: '.format(arg.text, prompt))                    
                if not value:
                    value = default                    
                self.env[name] = value
        self.os_env.update(self.env)
                
    def _attr(self, e, names):
        """ Получение значения XML-атрибута @a name с учётом выставленных
            значений переменных.
        """
               
        if isinstance(names, basestring):
            value = e.get(names)
            return value.format(**self.os_env)
        elif isinstance(names, list):
            values = []
            for name in names:
                v = e.get(name)
                if v is None:
                    values.append(None)
                    #print 'ERROR name = ', name
                else:
                    values.append(v.format(**self.os_env))
            return dict(zip(names, values))            
    
    def _add_user(self, e):
        """ Добавление пользователей.
        """

        if not check_root():
            print >>stderr, 'Permission denied: need to be "root"'
            exit(1)
        
        name = e.get('name')
        if not name:
            print >>stderr, 'User name MUST be supplied'
            exit(1)
            
        password = e.get('password')
        home = e.get('home', '/'.join(['/home', name]))
        sudoer = (e.get('sudoer', 'false').lower() == 'true')        
        add_user(name, password, home, sudoer)
    
    def _install_packages(self, tree):
        """ Установка пакетов разных типов.
        """
        
        if not check_root():
            print >> stderr, 'Permission denied: need to be "root"'
            exit(1)
        
        self._install_apt(tree.findall('Apt'))
        self._install_pip(tree.findall('Pip'))        
    
    def _install_apt(self, tree_list):
        """ Установка всех APT-пакетов.
        """

        if len(tree_list) == 0:
            return

        print "Installing APT packages."        

        repos = []
        packages = []
        
        for p in tree_list:
            try:
                r = {'repo': p.find('Repo').text, 'key': p.find('Key').text}
                if r['repo']:        
                    repos.append(r)
            except:
                pass                          
            packages.append(p.attrib)
        
        for r in repos:
            add_apt_repos(repos)
        for p in packages:
            install_apt_package(**p)
            
    def _install_pip(self, tree_list):
        """ Установка всех PIP-пакетов.
        """
        
        for p in tree_list:
            install_pip_package(**p.attrib)
            
    def _deploy_file(self, f):
        """ Копирование файла/директории из инсталляционного пакета.
        """     
        
        copy_file(f.get('src'), f.get('dest'))
        
    def _run_command(self, e):
        """ Выполнение произвольной shell-команды.
        """
        
        try:
            shell_run(e.text, env=self.env, cwd=e.get("pwd"))
        except CommandError, e:
            print >>stderr, "FAILED"
            print >>stderr, "Command:", e.cmd
            print >>stderr, "Output:", e.output
            exit(1)
            
    def _make_dirs(self, e):
        """ Создание каталога.
        """
        paths = [l.strip() for l in e.text.strip().splitlines()]
        mode = string.atoi(e.get('mode', '777'), base=8)
        make_dirs(paths, e.get('owner'), mode)
    
    def _download_bitbucket(self, e):
        """ Загрузка исходников с BitBucket'a.
        """
        args = self._attr(e, ['user', 'password', 'fork', 'repo', 'rev'])
        download_bitbucket(**args)

    def _append(self, e):
        """ Добавление строки к файлу
        """
        if (e.text != None):
            text = "\n".join([l.strip() for l in e.text.splitlines()])
            unique = (e.get("unique") == "True")
            file_name = e.get("file")
            append_to_file(file_name, text, unique)
            
    def _clone_mercurial(self, e):
        """ Клонирование Hg-репозитория.
        """
        args = self._attr(e, ['repo', 'rev'])
        mercurial_clone(**args)
        
    def _http_download(self, e):
        """ Загрузка файла по HTTP.
        """
        args = self._attr(e, ['url', 'user', 'password'])
        http_download(**args)


if __name__ == '__main__':
    from sys import argv
    
    if len(argv) > 1:
        cfg = XMLConfigurator()
        cfg.run_xml(argv[1])
    else:
        print "Usage: python {0} CONFIG_FILE".format(__file__)
