'''
    Copyright (c) Supamonks Studio and individual contributors.
    All rights reserved.

    This file is part of kabaret, a python Digital Creation Framework.

    Kabaret is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.
        
    Redistributions in binary form must reproduce the above copyright 
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.
    
    Kabaret is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.
    
    You should have received a copy of the GNU Lesser General Public License
    along with kabaret.  If not, see <http://www.gnu.org/licenses/>

--

    The kabaret.core.admin.projects.project module:
        Defines the Project class.
    
'''
import os
import sys
import inspect 
import platform

from kabaret.core.utils.importlib import resolve_ref
from kabaret.core.mq.name_service import ExposeServices
from kabaret.core.mq.zmq_rpc import RPCService, rpc_method

import kabaret.core.apps.base

from . import shape
from . import settings

ENV__STATION_CLASS = 'KABARET_STATION_CLASS'

class ProjectError(Exception):
    pass

class ProjectRPC(RPCService):
    def __init__(self, project):
        super(ProjectRPC, self).__init__()
        self.project = project
        
    @rpc_method
    def settings(self):
        return self.project.settings

    @rpc_method
    def test(self):
        return 'OK TEST', repr(self.project.settings.STATIONS.CLASSES)
    
import multiprocessing
class ProjectRPCProcess(multiprocessing.Process):
    def __init__(self, name, project):
        super(ProjectRPCProcess, self).__init__(name=name)
        self.project = project
        self.daemon = True
        
    def run(self):
        service = ProjectRPC(self.project)
        service.bind('tcp://*:9997')
        service.loop.start()
        
class Project(object):
    def __init__(self, store_path, name):
        self.store_path = store_path
        self.name = name

        self._settings = None   # will become a core.admin.projects.settings.ProjectSettings()
        self._station_class_name = None
        self._station_class = None
        self._apps = None       # will become {} of app_key -> app
        
        self.name_service = None
    
    def _load_settings(self):
        # use the default project shape to find
        # the SETTINGS:
        s = shape.get()(self.store_path, self.name)
        settings_path = s.path('SETTINGS')
        self._settings = settings.ProjectSettings()
        self._settings.load(settings_path, project=self)

        my_station_class = None
        default_class = None
        for station_class in self._settings.STATIONS.CLASSES:
            if default_class is None and station_class.class_name is None:
                default_class = station_class
            elif station_class.class_name == self.station_class_name:
                my_station_class = station_class
                break
        if my_station_class is None:
            if default_class is None:
                raise ProjectError(
                    'Unable to find a StationClass matching %r, and no default found either'%(
                        self.station_class_name
                    )
                )
            my_station_class = default_class
        self._station_class = my_station_class
        
        for path in self._station_class.python_paths:
            if path not in sys.path:
                sys.path.insert(0, path)
                
    @property    
    def settings(self):
        if self._settings is None:
            self._load_settings()
        return self._settings

    @property
    def station_class_name(self):
        if self._station_class_name is None:
            self._cache_station_class_name()
        return self._station_class_name
    
    def _cache_station_class_name(self):
        if ENV__STATION_CLASS in os.environ:
            station_class_name = os.environ[ENV__STATION_CLASS]
        else:
            station_class_name = platform.platform(terse=True)
        self._station_class_name = station_class_name
        
    def _load_apps(self):
        self._apps = {}
        app_refs = self.settings.APPS
        for app_ref in app_refs:
            try:
                app = resolve_ref(app_ref)
            except ValueError:
                raise ProjectError('Bad app ref in SETTINGS.APPS: %r'%(app_ref,))
            except ImportError:
                raise ProjectError(
                    'Unable to import module for app ref %r. '
                    'You may want to use affine your STATIONS.CLASSES in the project SETTINGS.'%(
                        app_ref
                    )
                )
            except AttributeError:
                raise ProjectError(
                    'Unable to find the class %r '%(app_ref,)
                )

            if not inspect.isclass(app) or not issubclass(app, kabaret.core.apps.base.App):
                raise ProjectError(
                    'The class %r is not a kabaret.core.apps.base.App'%(app_ref,)
                )
            
            self._apps[app.APP_KEY] = app(self)
            
                
    @property
    def apps(self):
        '''
        A dict of app_key: app_instance
        '''
        if self._apps is None:
            self._load_apps()
        return self._apps
    
    def get_app(self, app_key):
        '''
        Returns the app declared with the given key, or
        None if no such app was declared in the project
        settings.
        '''
        return self.apps.get(app_key)
    
    def start_serve(self):
        '''
        Setup project message queues and name service for
        each connection points.
        '''
        if self.name_service is not None and self.name_service.is_alive():
            print 'Already Serving'
            return
        
        self.project_rpc = ProjectRPCProcess('ProjectRPC_'+self.name, self)
        self.project_rpc.start()
        
        services = {
            'reversion': 'http://the_url:reversion_port',
            'project:%s'%(self.name,): 'http://the_url:project_port',
        }
        self.name_service = ExposeServices(
            name="Kabaret_project_NameServer:%s"%(self.name,),
            frequence=1, exposure_port=self.settings.NAME_SERVICE.request_port,
            **services
        )
        self.name_service.daemon = True
        self.name_service.start()
        
    def stop_serve(self):
        if self.name_service is None or not self.name_service.is_alive():
            print 'Not Serving'
            return
        self.name_service.stop()
        