'''
    Copyright (c) Supamonks Studio and individual contributors.
    All rights reserved.

    This file is part of kabaret, a python Digital Creation Framework.

    Kabaret is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.
        
    Redistributions in binary form must reproduce the above copyright 
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.
    
    Kabaret is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.
    
    You should have received a copy of the GNU Lesser General Public License
    along with kabaret.  If not, see <http://www.gnu.org/licenses/>

--

    The kabaret.core.mq.name_service:
        Defines the ExposeServices and the ServiceFinder.
          
'''


'''



'''
import sys
import time
import multiprocessing
import socket, select

class ExposeServices(multiprocessing.Process):
    '''
    This Process can be run to provide service information in
    response to a service call from the find_service() function.
    
    '''
    def __init__(self, name, frequence=1, exposure_port=9999, **services):
        '''
        The services kwargs must be in the form:
            service_name: endpoint
            
        Typical Usage:
            exposer = ExposeServices()
            exposer.start()
            ...
            exposer.stop() # blocks until stopped
        '''
        super(ExposeServices, self).__init__(name=name)
        
        self.stop_event = multiprocessing.Event()
        self.frequence = frequence
        self.services = services
        
        self.listening_port = exposure_port
        self.listening_socket = None
        
        self.broadcast_socket = None
        
    def log(self, *args):
        sys.stderr.write(
            '{}: {}{}'.format(
                multiprocessing.current_process().name, 
                ' '.join([ str(i) for i in args ]),
                '\n'
            )
        )
        sys.stderr.flush()

    def _setup_listening(self):
        self.log('Starting Service(s) Exposure: Listener.')
        self.listening_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.listening_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)

        bound = False
        nb_tries = 5
        for i in range(nb_tries):
            try:
                self.listening_socket.bind(('', self.listening_port))
                bound = True
            except socket.error, err:
                self.log('  ' , str(err))
                self.log('Connection failed (%i/%i). Trying again in 1sec...\n'%(i, nb_tries))
                time.sleep(self.frequence)
            else:
                break
            
        if not bound:
            self.log('Could not connect. Aborting')
            raise RuntimeError('Could not connect. Aborting')

    def _setup_broadcast(self):
        self.log('Starting Service(s) Exposure: Responder.')
        self.broadcast_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.broadcast_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        self.broadcast_socket.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
        self.broadcast_socket.bind(('', 0))

    def stop(self):
        self.log('Stopping Service Exposure.')
        self.stop_event.set()
        self.join()
        self.log('Service Exposure Stopped.')
        
    def run(self):
        self._setup_listening()
        self._setup_broadcast()

        while True:
            if self.stop_event.wait(self.frequence):
                print 'Stop Service Exposure requested.'
                return
            
            #self.log('Listening...')
            
            ready = select.select([self.listening_socket], [], [], 1)
            if ready[0]:
                data, _wherefrom = self.listening_socket.recvfrom(1024, 0)
                try:
                    if not data.startswith('service_lookup:'):
                        raise AttributeError
                except AttributeError:
                    self.log('Got broadcasted garbage: %r %s\n'%(data, data))
                    continue
                
                try:
                    service_name, reply_port = data.split(':', 1)[1].split(' ', 1)
                    reply_port = int(reply_port)
                except:
                    self.log('Got malformed service lookup: %r\n'%(data,))
                    continue
                    
                self.log('Got request for service: %r\n'%(service_name,))
                endpoint = self.services.get(service_name, None)
                if endpoint is None:
                    self.log('  Unknown service.\n')
                    continue
                
                self.log('  Broadcasting service endpoint: %r'%(endpoint,))
                broadcast_message = 'service_endpoint:%s %s'%(service_name, endpoint,)
                self.broadcast_socket.sendto(broadcast_message, ('<broadcast>', reply_port))


#class ServiceFinder(multiprocessing.Process):
#    '''
#    
#    '''
#    def __init__(self, service_name, nb_tries=1, timeout=4, request_port=9999, response_port=9998):
#        '''
#        Typical Usage:
#            finder = ServiceFinder('service_name')
#            exposer.start()
#            ...
#            exposer.stop() # blocks until stopped
#        '''
#        super(ExposeServices, self).__init__()
#        
#        self.stop_event = multiprocessing.Event()
#        self.result_queue = multiprocessing.Queue()
#
#        self.service_name = service_name
#        self.nb_tries = nb_tries
#        self.timeout = timeout
#        
#        self.request_port = request_port
#        self.broadcast_socket = None
#
#        self.response_port = response_port
#        self.listening_socket = None
#
#        self.broadcast_message = 'service_lookup:%s %i'%(self.service_name, self.listening_port)
#
#    def _setup_listening(self):
#        self.log('Starting Service Finder: Listener.')
#        listening_port = self.response_port
#        self.listening_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
#        self.listening_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
#        
#        bound = False
#        bind_nb_tries = 5
#        for i in range(bind_nb_tries):
#            try:
#                self.listening_socket.bind(('', listening_port))
#                bound = True
#            except Exception, err:
#                print str(err)
#                print 'Listening bind failed (%i/%i). Trying again in 1sec...'%(i, bind_nb_tries)
#                time.sleep(1)
#            else:
#                break
#                
#        if not bound:
#            print 'Could not bind a socket for listening.'
#            return
#
#    def _setup_broadcast(self):
#        self.log('Starting Service Finder: Requester.')
#        self.broadcast_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
#        self.broadcast_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
#        self.broadcast_socket.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
#        self.broadcast_socket.bind(('', 0))
#
#    def stop(self):
#        self.log('Stopping Service Finder.')
#        self.stop_event.set()
#        self.join()
#        self.log('Service Finder Stopped.')
#        return self.result_queue.get()
#    
#    def accept_service(self, service_name):
#        '''
#        Returns True if the given service_name matches the one
#        we look for.
#        '''
#        return service_name == self.service_name
#    
#    def run(self):
#        self._setup_listening()
#        self._setup_broadcast()
#
#        # Broadcast a service lookup:
#        print "Calling for Service", self.service_name
#        self.broadcast_socket.sendto(self.broadcast_message, ('<broadcast>', self.request_port))
#        
#        i = 0
#        while True:
#            if i > self.nb_tries or self.stop_event.wait(0):
#                print 'Stop Service Exposure requested.'
#                break
#            
#            # Listen for a response:
#            listening_ready = select.select([self.listening_socket], [], [], self.timeout)
#            if not listening_ready[0]:
#                print 'Nothing found...'
#                i += 1
#                continue
#            
#            data, _wherefrom = self.listening_socket.recvfrom(1024, 0)
#            try:
#                if not data.startswith('service_endpoint:'):
#                    raise AttributeError
#            except AttributeError:
#                sys.stderr.write('Got broadcasted garbage: %r %s\n'%(data, data))
#                sys.stderr.flush()
#                continue
#            
#            try:
#                service_name, sent_endpoint = data.split(':', 1)[1].split(' ', 1)
#            except:
#                sys.stderr.write('Got malformed project server: %r\n'%(data,))
#                sys.stderr.flush()
#                continue
#            
#            if self.accept_service(service_name): 
#                sys.stderr.write('Got unrequested service: %r\n'%(data,))
#                sys.stderr.flush()
#                continue
#            
#            endpoint = sent_endpoint
#            sys.stderr.write('Got service endpoint: %s %s\n'%(service_name, endpoint))
#            sys.stderr.flush()
#            self.result_queue.put(endpoint)
#            break
        
def find_service(service_name, nb_tries=1, timeout=4, exposure_port=9999, response_port=9998):
    '''
    Returns the endpoint of the specified service.
    '''
    # SETUP LISTENING
    listening_port = response_port
    listening_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    listening_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
    
    bound = False
    bind_nb_tries = 5
    for i in range(bind_nb_tries):
        try:
            listening_socket.bind(('', listening_port))
            bound = True
        except Exception, err:
            print str(err)
            print 'Listening bind failed (%i/%i). Trying again in 1sec...'%(i, bind_nb_tries)
            time.sleep(1)
        else:
            break
            
    if not bound:
        print 'Could not bind a socket for listening.'
        return

    # SETUP BROASDCAST
    broadcast_port = exposure_port
    broadcast_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    broadcast_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
    broadcast_socket.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
    broadcast_socket.bind(('', 0))
    broadcast_message = 'service_lookup:%s %i'%(service_name, listening_port)
    
    # Broadcast a service lookup:
    print "Calling for Service", service_name
    broadcast_socket.sendto(broadcast_message, ('<broadcast>', broadcast_port))
    
    endpoint = None
    i = 0
    while True:
        if i > nb_tries:
            break
        
        # Listen for a response:
        listening_ready = select.select([listening_socket], [], [], timeout)
        if not listening_ready[0]:
            print 'Nothing found...'
            i += 1
            continue
        
        data, _wherefrom = listening_socket.recvfrom(1024, 0)
        try:
            if not data.startswith('service_endpoint:'):
                raise AttributeError
        except AttributeError:
            sys.stderr.write('Got broadcasted garbage: %r %s\n'%(data, data))
            sys.stderr.flush()
            continue
        
        try:
            sent_service_name, sent_endpoint = data.split(':', 1)[1].split(' ', 1)
        except:
            sys.stderr.write('Got malformed project server: %r\n'%(data,))
            sys.stderr.flush()
            continue
        
        if sent_service_name != service_name: 
            sys.stderr.write('Got unrequested service: %r\n'%(data,))
            sys.stderr.flush()
            continue
        
        endpoint = sent_endpoint
        sys.stderr.write('Got service endpoint: %s %s\n'%(service_name, endpoint))
        sys.stderr.flush()
        break
    
    if endpoint is not None:
        print 'FOUND SERVICE:', service_name, endpoint
    return endpoint


def main_a():
    exp = ExposeServices(s1='tcp:\\s1:12345', s2='tcp:\\s2:12346')
    exps = [exp]
    running = False
    while True:
        a = raw_input('Command (s or q):')
        if a.startswith('n '):
            sn = a[2:].strip()
            new_exp = ExposeServices(**{sn:'ENPOINT(%r)'%(sn,)})
            new_exp.start()
            exps.append(new_exp)
            
        elif a == 's':
            if exp.is_alive():
                print 'Service is already running'
                continue
            try:
                exp.start()
            except:    
                import traceback
                traceback.print_exc()
                break
            
        elif a == 'q':
            for an_exp in  exps:
                if an_exp.is_alive(): 
                    print 'Stopping service', an_exp.name
                    try:
                        an_exp.stop()
                    except:
                        import traceback
                        traceback.print_exc()
            print 'Quitting'
            break
    
    raw_input('Enter to close:')

def main_b():
    service_name = 'project:TEST'
    find_service(
        service_name, nb_tries=1, timeout=4, exposure_port=9999, response_port=9998
    )

def main_c():
    service_name = 'apphost:TEST'
    find_service(
        service_name, nb_tries=3, timeout=4, exposure_port=9999, response_port=9998
    )
    
if __name__ == '__main__':
    main_c()