
import select
import contextlib
import Pyro4

from kabaret.core.events.dispatcher import EventDispatcher
#from kabaret.core.utils import opi
from . import url
from . import CommunicationError

class ClientService(object):
    def __init__(self, client):
        self._client = client
        
    def receive_event(self, event):
        self._client.receive_event(event)


class CommandTrigger(object):
    def __init__(self, to_trigger, path=[]):
        super(CommandTrigger, self).__init__()
        self.to_trigger = to_trigger
        self.path = path

    def __getitem__(self, name):
        return getattr(self, name)
    
    def __getattr__(self, name):
        return CommandTrigger(
            to_trigger=self.to_trigger,
            path = self.path+[name]
        )

    def __call__(self, *args, **kwargs):
        return self.to_trigger(self.path, *args, **kwargs)
        
class Namespace(object):
    def __init__(self):
        super(Namespace, self).__init__()
        self._names = set()
    
    def __getitem__(self, name):
        return getattr(self, name)
    
    def add(self, name, o):
        setattr(self, name, o)
        self._names.add(name)
    
    def clear(self):
        for name in self._names:
            delattr(self, name)
            
class AppGetter(Namespace):
    # this is only for better AttributeError message
    # when the app key is unknown
    pass

class ProjAppGetter(Namespace):
    # this is only for better AttributeError message
    # when the app key is unknown
    pass

class Client(object):
    def __init__(self, project_name):
        super(Client, self).__init__()
        
        self.project_name = project_name
        
        self._daemon = None
        self._backservice = None
        self._init_backservice()
        
        self.apphost = None
        self.async_apphost = None
        
        self._current_future_target = None # contextual result handler
        self._no_result = False # context for one way calls
        self._futures = set()
        
        self.apps = AppGetter()         # Client apps commands 
        self.project_apps = ProjAppGetter() # Project apps commands 
        
        self._event_dispatcher = EventDispatcher()
        self.connect()
    
    def _init_backservice(self):
        self._backservice = ClientService(self)
        self._daemon = Pyro4.Daemon()
        self._daemon.register(self._backservice)

    def shutdown(self):
        print 'Shutting down client' 
        self.disconnect()
        self._daemon.shutdown()
        self._daemon = None
        
    def disconnect(self):
        if self.apphost is not None:
            self.apphost._pyroRelease()
            self.apphost = None
            self.async_apphost = None
        self._command_triggers = []
        
    def connect(self):
        self.disconnect()
        self.apphost = url.resolve(url.For.apphost(self.project_name), local=True, ping=True)
        self.async_apphost = Pyro4.async(self.apphost)
        self.apphost._pyroOneway.add('register_client')
        self.apphost.register_client(self._backservice)
        
        self._set_command_triggers()
    
    def _clear_command_triggers(self):
        self.apps.clear()
        self.project_apps.clear()
            
    def _set_command_triggers(self):
        for key in self.apphost.app_keys():
            ct = CommandTrigger(self.trigger_app_action, [key])
            self.apps.add(key, ct)
            
        for key in self.apphost.app_keys(in_project=True):
            ct = CommandTrigger(self.trigger_project_app_action, [key])
            self.project_apps.add(key, ct)
    
    def kill_apphost(self):
        self.apphost._pyroOneway.add('kill')
        self.apphost.kill()
        
    def tick(self):
        #print 'Ticking the app host'
        if self.apphost is None:
            return 'Disconnected'
        
        while True:
            # flush all pyro socket events (callbacks from self.apphost)
            s,_,_ = select.select(self._daemon.sockets,[],[],0.01)
            if s:
                self._daemon.events(s)
            else:
                break

        remaining_futures = set()
        for f in self._futures:
            print f
            if f.ready:
                try:
                    value = f.value
                except CommunicationError:
                    # all the futures are lost and
                    # would raise too, we need to
                    # drop them and let the user
                    # reconnect.
                    self._futures = set()
                    self.disconnect()
                    return 'Disconnected' 
                except:
                    import sys
                    Pyro4.util.excepthook(*sys.exc_info())
                    continue
                print 'FUTURE READY', value
                try:
                    f._on_result(f.value)
                except Exception, err:
                    self.on_result_error(f, err)
                    raise
            else:
                remaining_futures.add(f)
        self._futures = remaining_futures
        return 'Q:'+str(len(remaining_futures))
    
    def trigger_project_app_action(self, path, *args, **kwargs):
        return self._trigger_apphost_cmd('do_project_cmd', path, args, kwargs)

    def trigger_app_action(self, command_path_, *args, **kwargs):
        #NB: this disallow the use of a 'command_path_' argument in
        # the actual triggered command...
        return self._trigger_apphost_cmd('do_cmd', command_path_, args, kwargs)
        
    def _trigger_apphost_cmd(self, apphost_cmd_name, path, args, kwargs):
        if self._current_future_target is not None:
            if self._no_result:
                raise Exception('Cannot be async and one way! (result_to + no_result)')
            future = getattr(self.async_apphost, apphost_cmd_name)(
                path, args, kwargs
            )
            future._on_result = self._current_future_target
            self._futures.add(future)
            return future
        else:
            if self._no_result:
                self.apphost._pyroOneway.add('do_cmd')
            else:
                try:
                    self.apphost._pyroOneway.remove('do_cmd')
                except KeyError:
                    pass
            ret = getattr(self.apphost, apphost_cmd_name)(
                path, args, kwargs
            )
            return ret
        
    @contextlib.contextmanager
    def no_result(self):
        self._no_result = True
        yield
        self._no_result = False
        
    @contextlib.contextmanager
    def result_to(self, callable):
        previous_future_target = self._current_future_target
        self._current_future_target = callable
        yield
        self._current_future_target = previous_future_target
        
    def receive_event(self, event):
        print 'Client', self, 'got event:', event
        self._event_dispatcher.dispatch(event)
        if event.etype == event.TYPE.MESSAGE:
            self.on_message(event.data)
            
    def add_event_handler(self, handler, path, etype=None):
        self._event_dispatcher.add_handler(
            handler, path, etype
        )

    def send_event(self, event):
        self._event_dispatcher.dispatch(event)
        
    def get_commands(self, app_key=None, menu=None, sync=False):
        '''
        Returns the ui information of the commands in the 
        given app (or all apps) that are present
        in the given menu (or all menus).
        
        if sync is True, the call will block and return
        the result directly.
        ''' 
        if sync:
            return self.apphost.get_commands(app_key, menu)
        #else:
        future = self.async_apphost.get_commands(app_key, menu)
        future._on_result = self._current_future_target
        self._futures.add(future)
    
#    def action(self, arg):
#        self._futures.add(self.async_apphost.action(arg))
#    
#    def a(self):
#        self.apphost.a('THIS IS A TEST')

    def on_result_error(self, future, error):
        print 'ERROR while getting future result:\n  %s'%(error,)
    
    def on_result(self, result):
        print 'on_result', result 

    def on_message(self, msg):
        print 'CLIENT MESSAGE:', msg
        