'''
    Copyright (c) Supamonks Studio and individual contributors.
    All rights reserved.

    This file is part of kabaret, a python Digital Creation Framework.

    Kabaret is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.
        
    Redistributions in binary form must reproduce the above copyright 
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.
    
    Kabaret is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.
    
    You should have received a copy of the GNU Lesser General Public License
    along with kabaret.  If not, see <http://www.gnu.org/licenses/>

--

    The kabaret.core.admin.projects.settings module:
        Defines the project settings and their defaults.
    
'''
import pprint 

from kabaret.core.conf import Config, Group, Attribute
 
class StationClass(object):
    def __init__(self, class_name, store_path, python_paths):
        self.class_name = class_name
        self.store_path = store_path
        self.python_paths = python_paths
    
    def __repr__(self):
        return '%s(\n\tclass_name=%r,\n\tstore_path=%r,\n\tpython_paths=%s\n)'%(
            self.__class__.__name__,
            self.class_name, 
            self.store_path,
            pprint.pformat(self.python_paths)
        )
        
class ProjectSettings(Config):
    '''
    The Project Configuration.
    
    You can use 'project' to access the kabaret.core.admin.projects.project.Project
    instance that loads the file.
    For example: 
        store = project.store_path
        proj_name = project.name
    
    Beware that this project instance has no settings yet and thus cannot
    be very helpful.
    Also, the project.store_path is the server one and is likely to be different
    from yours.
    
    '''
    EXTRA_CONTEXT = {
        'StationClass': StationClass,
    }
    
    def _define(self):
        self.SHAPE = Group(
            'The shape defines the mandatory filesystem structure of the project.',
            custom_loader = Attribute(
                None, 
                'An entry point for custom shapes. '
                'Use a callable with no argument.'
            ),
            name = Attribute(None, 'The shape name. Use None for kabaret default') 
        )
        
        self.NAME_SERVICE = Group(
            '''
            The name service is used to discover servives on the LAN.
            You can tweak the ports to avoid (rare) clashes with other
            softwares using broadcasting UDP. Beware that all clients must
            know those ports number''',
            request_port = 9999,
            reponse_port = 9998
        )
        
        self.STATIONS = Group(
            '''
            This section let you define the different class
            of stations used in the project.
            
            You must instantiate some StationClass and 
            store then in the STATIONS.CLASSES:
            
            win7_work = StationClass(
                class_name='Windows-7',
                store_path='path/to/store',
                python_paths=['more/path', 'for/python'],
            )
            
            The 'class_name' must be a string identifying the
            station class. 
            On the other end, the station class is read
            from the KABARET_STATION_CLASS env variable, or
            the value of "platform.platform(terse=True)" if the 
            variable is not present.
            You 'class_name' must be defined accordingly to this.
            
            A StationClass with None as class_name may be
            defined to be used when no matching class can
            be found.
            
            ''',
            CLASSES = [
                StationClass(
                    None, 
                    store_path="path/to/store",
                    python_paths=[]
                )
            ]
        )
        
        self.APPS = Attribute(
           [],
           '''
           List of apps you want to allow in the project.
           Each entry must be a string of the app package path, i.e: kabaret.workstation.apps.flow
           '''
        )        