

import inspect

from kabaret.core.utils import importlib
from .base import App

class AppError(Exception):
    pass

class AppLoadError(AppError):
    pass

def load_apps(app_refs, project=None, apphost=None):
    if project is None and apphost is None:
        raise ValueError(
            'You must require apps for a project or for an apphost'
        )
    
    apps = {}
    for app_ref in app_refs:
        try:
            app = importlib.resolve_ref(app_ref)
        except ValueError:
            raise AppLoadError('Bad app ref: %r'%(app_ref,))
        except ImportError:
            raise AppLoadError(
                'Unable to import module for app ref %r. '
                'You may want to use affine your STATIONS.CLASSES in the project SETTINGS.'%(
                    app_ref
                )
            )
        except AttributeError:
            raise AppLoadError(
                'Unable to find the class %r '%(app_ref,)
            )

        if not inspect.isclass(app) or not issubclass(app, App):
            raise AppLoadError(
                'The class %r is not a %s.%s'%(app_ref, App.__module__, App.__name__)
            )
        
        if project is not None and app.IS_PROJECT_APP:
            apps[app.APP_KEY] = app(host=project)
        if apphost is not None and app.IS_CLIENT_APP:
            apps[app.APP_KEY] = app(host=apphost)
    
    return apps
