'''
    Copyright (c) Supamonks Studio and individual contributors.
    All rights reserved.

    This file is part of kabaret, a python Digital Creation Framework.

    Kabaret is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.
        
    Redistributions in binary form must reproduce the above copyright 
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.
    
    Kabaret is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.
    
    You should have received a copy of the GNU Lesser General Public License
    along with kabaret.  If not, see <http://www.gnu.org/licenses/>

--

    The kabaret.core.apps.base package:
        Defines the App abstract class.
    
'''
import inspect 

from kabaret.core import conf
from .command import Command

class LinkedAppError(Exception):
    pass

class AppType(type):
    '''
    App class type.
    
    Generates the App._api
    
    '''
    def __new__(cls, class_name, bases, class_dict):
        # Build the REQ_APPS descriptors:
        req_app_descriptors = {}
        req_app_keys = class_dict.get('REQ_APPS', [])
        for app_key in req_app_keys:
            if app_key in class_dict:
                #TODO: also lookup bases attributes
                raise LinkedAppError(
                    "Cannot link to app %r: a class attribute has this name"%(
                        app_key
                    )
                )
            req_app_descriptors[app_key] = ReqAppDescriptor(app_key)
            
        # Build api
        commands = {}
        for n, o in class_dict.items():
            #print class_name, n
            #print '   ', o, dir(o)
            if inspect.isclass(o) and issubclass(o, Command):
                class_dict[n] = cls.make_command_func(n, o)
                commands[n] = o
                                
        # augment the class_dict
        class_dict.update(req_app_descriptors)
        class_dict['_commands'] = commands
        
        # now create the class
        app_class = super(AppType, cls).__new__(cls, class_name, bases, class_dict)
        return app_class

    @staticmethod
    def make_command_func(name, cmd_class):
        def trigger_app_command(app, *args, **kwargs):
            if args:
                raise TypeError(
                    'Command must use keywork arguments!\n  %s\n  got: %s, %s,'%(
                        cmd_class.usage(), args, kwargs
                    )
                )
            try:
                cmd = cmd_class(app, **kwargs)
            except TypeError, err:
                message = err.message+'\n  %s\n  got:'%(cmd_class.usage(), kwargs)
                raise TypeError(message)
            ret = cmd.doit()
            return ret
        return trigger_app_command
    
class API(object):
    def __init__(self, actions):#, menus):
        super(API, self).__init__()
        self.actions = actions
        #self.menus = menus
    
#    def to_dict(self):
#        return {
#            'actions':[a.to_dict() for a in self.actions ],
#            #'menus':[m.to_dict() for m in self.menus ],
#        }
#    
#    def from_dict(self, d):
#        self.actions = [ AppAction.from_dict(ad) for ad in d['actions'] ]
    
    def get_menus(self):
        all_names = []
        [ all_names.extend(a.menus) for a in self.actions.values() ]
        return set(all_names)
    
    def get_actions(self, menu, blocked_topics=[]):
        '''
        Return all the actions of the requested menus
        with topic not listed in blocked_topics.
        
        If menu is None, all menus are returned.
        '''
        return [
            a for a in self.actions.values()
            if (menu is None or menu in a.menus) and not [ True for t in a.topics if t in blocked_topics ]
        ]
            
class ReqAppDescriptor(object):
    def __init__(self, app_key):
        super(ReqAppDescriptor, self).__init__()
        self.app_key = app_key
    
    def __get__(self, app, apptype):
        if app is None:
            raise UnboundLocalError('Cannot get req app without app instance')
        if self.app_key not in app._req_apps:
            req_app = app.host.app(self.app_key)
            if req_app is None:
                raise LinkedAppError(
                    'Cannot find app %r required by app %r'%(
                        self.app_key, app.APP_KEY
                    )
                )
            return req_app

class AppAction(object):
    def __init__(self, name=None, topics=[], blocking_for=[], label=None, menus=[], icon=None):
        self.name = name
        self.topics = topics
        self.blocking_for = blocking_for
        self.label = label
        self.menus = menus
        self.icon = icon
    
    def to_dict(self):
        return {
            'name':self.name,
            'topics':self.topics,
            'blocking_for':self.blocking_for,
            'label':self.label,
            'menus':self.menus,
            'icon':self.icon,
        }
    
    @classmethod
    def from_dict(cls, d):
        return cls(**d)

#class AppActionMenu(object):
#    def __init__(self, name):
#        self.name = name
#        self.actions = []
    
class App(object):
    '''
    An App defines a set of actions to do in the context
    of a project.
    
    Is has a unique APP_KEY (class attribute)

    Each class defines a REQ_APPS list containing the 
    keys of other apps that are required (used) by this one.
    In the App instance, those required App will be accessible
    thru the attribute named like the required app key, i.e:
        class MyApp(App):
            REQ_APPS = ['BACKEND']
        ...
        app = MyApp(host)
        app.BACKEND.get_infos()
        
    All apps of a project are instantiated at startup time
    so you should avoid doing much in the app constructor.
    It is a good practice to init your class attributes
    with None and only load them at the first access
    (or after some kind of cache flush)
    
    To turn an App method into an action, use the App.action
    decorator.
    
    '''
    __metaclass__ = AppType
    _api = None
    
    APP_KEY = None
    REQ_APPS = []

    IS_PROJECT_APP = False
    IS_CLIENT_APP = True

    @staticmethod
    def action(name=None, topics=[], blocking_for=[], label=None, menus=[], icon=None):
        '''
        Make a method of an App available to remote users and in GUI Menus.
        
        '''
        action = AppAction(name, topics, blocking_for, label, menus, icon)
        def action_decorator(decorable, action=action):
            decorable._kabaret_action = action
            if action.name is None:
                action.name = decorable.func_name
            return decorable
        return action_decorator
    
    @staticmethod
    def query(name=None, topics=[], blocking_for=[], label=None, menus=[], icon=None):
        '''
        Make a method of an App available to remote users and in GUI Menus.
        
        '''
        #TMP:
        return App.action(name, topics, blocking_for, label, menus, icon)
    
        action = AppAction(name, topics, blocking_for, label, menus, icon)
        def query_decorator(decorable, action=action):
            decorable._kabaret_query = action
            if action.name is None:
                action.name = decorable.func_name
            return decorable
        return query_decorator
    
    def __init__(self, host):
        super(App, self).__init__()

        self.host = host
        self._req_apps = {}  # api_key -> app
        
        self._event_emitter = None
        self._event_handler_adder = None
        
    def set_event_emitter(self, callable):
        '''
        Called by the host supporting this app.
        '''        
        self._event_emitter = callable
    
    def emit_event(self, event):
        '''
        Emits an app event.
        Handlers can be one of the other apps
        or anything in the client.
        '''
        if self._event_emitter is not None:
            self._event_emitter(event)
    
    def set_event_handler_adder(self, callable):
        '''
        Called by the host supporting this app.
        '''
        self._event_handler_adder = callable
    
    def add_app_event_handler(self, handler, path, etype):
        '''
        Registers an event handler for the given path and 
        etype.
        '''
        if self._event_handler_adder is None:
            raise RuntimeError('No app event handler adder was defined')
        self._event_handler_adder(handler, path, etype)

    def _load_settings(self, key, path):
        '''
        Tells the application to load the settings 
        in the folder 'path'.
        You must override _load_settings and not
        this one to let your App configure itself.
        '''
        print '#--- LOADING APP SETTINGS', self.APP_KEY
        settings_class = self._get_setting_class()
        settings_context = self._get_setting_context()
        if settings_class is None:
            self._apply_settings(None)
            return
        settings = settings_class()
        settings_file = '%s/%s.kbr'%(path, key)
        try:
            settings.load(settings_file, **settings_context)
        except conf.ConfigMissingError:
            # Save the default config so that
            # admin can edit it.
            settings.save(settings_file)
            
        except conf.ConfigError:
            import traceback
            traceback.print_exc()
            # using default settings.
            pass
        
        self._apply_settings(settings)
        
    def _get_setting_context(self):
        '''
        Returns the context to use when
        reading the settings file.
        See kabaret.core.conf.Config.load()
        
        The default implementation returns
        an empty dict.
        '''
        return {}
    
    def _get_setting_class(self):
        '''
        Returns the class to use for the App
        settings.
        The returned class must be a sub-class
        of kabaret.core.conf.Config that does
        not need argument in its constructor.
        
        If the App does not save/read settings, 
        None should be returned (which is the 
        default behavior)
        
        '''
        return None
    
    def _apply_settings(self, settings):
        '''
        Called when the App settings are ready to 
        be applied.
        
        If the class does not use settings (by
        overriding _get_settings_class), the 
        settings argument will be None
        
        The default implementation does
        nothing.
        '''
        pass

    def _host_init_done(self):
        '''
        Called when the AppHost has finished
        loading all apps.
        You can use this placeholder to do 
        app initialization that rely on other
        applications.
        
        The default implementation does nothing.
        '''
        pass
    
    

        