

#import logging 
import multiprocessing

import kabaret.core.admin.projects
from kabaret.core.events.event import Event
from kabaret.core.events.dispatcher import EventDispatcher
from kabaret.core.mq.async import AsyncService, AsyncClient
from kabaret.core.mq.name_service import ExposeServices, find_service


class PUB_TOPICS:
    EVENT = 'events'
    
class APIRPC(object):
    '''
    This helper class defines a namespace for app's
    actions.
    '''
    def __init__(self, service, app_name):
        super(APIRPC, self).__init__()
        self.service = service
        self.app_name = app_name
        
        app = service._project.apps[app_name]
        for name, action in app._api.actions.iteritems():
            print 'API', app_name, name, action
            setattr(self, name, self._create_action_handler(app, action))

    @staticmethod
    def _create_action_handler(app, action):
        # this is used to consolidate the 'app' and 'action' 
        # names used in the wrapper:
        def action_handler(*args, **kwargs):
            return action.callable(app, *args, **kwargs)
        return action_handler
    
class AppHostService(AsyncService):
    
    def __init__(self, project_service_address, project_service_port, project_service_protocol):
        super(AppHostService, self).__init__()
        self._app_event_dispatcher = EventDispatcher()

        print 'Connecting to Project Server:', project_service_address, project_service_port
        import kabaret.core.services.project
        self.project_service = kabaret.core.services.project.ProjectClient(
            project_service_address, project_service_port
        )
        store_path = 'X:/TEST_STORE' #TODO: get it from project_service
        project_name = self.project_service.sync('name')
        
        project = kabaret.core.admin.projects.Project(
            store_path, project_name
        )
        self._project = project
        
        for name, app in self._project.apps.items():
            api_rpc = APIRPC(self, name)
            setattr(self, name, api_rpc)
    
            app.set_app_event_emitter(self.emit_app_event)
            app.set_event_handler_adder(self.add_app_event_handler)
            
        self._control_ops = {
            # declared here (AppHostService):
            'stop':self.stop,
            'app_names': self.app_names,
            'menus': self.menus,
            'actions': self.actions,
            
            # declared in base (AsyncService):
            'publish': self.publish,
            '_get_ps_port': self._get_ps_port,
        }
        
        #TODO: deal with the address:
        #self.bind('*', port)
        self.bind_to_random('*')
        
    def _get_rpc_handler(self, op):
        meth = self._control_ops.get(op)
        if meth is not None:
            return meth
        
        try:
            scope, meth = op.split('.')
        except:
            return None
        scope = getattr(self, scope, None)
        meth = getattr(scope, meth, None)
        return meth
    
    def add_app_event_handler(self, handler, path, etype=None):
        '''
        This gives each app in the project the opportunity
        to receive events from other app.
        '''
        self._app_event_dispatcher.add_handler(
            handler, path, etype
        )

    def emit_app_event(self, event):
        # dispatch event to apps first
        self._app_event_dispatcher.dispatch(event)
        
        # publish to clients if still needed:
        if not event.propagate:
            return
        self.publish(PUB_TOPICS.EVENT, event.to_dict())

    def get_exposure_port(self):
        return self.project_service.sync('request_port')
    
    def app_names(self):
        return sorted(self._project.apps.keys())

    def menus(self):
        menus = set()
        [ menus.update(app.get_menus()) for app in self._project.apps.values() ]
        return menus
    
    def actions(self, menu):
        return [
            [ a.to_dict() for a in app.get_actions(menu) ]
            for app in self._project.apps.values()
        ]
        
    def stop(self):
        self.loop.stop()
        
class AppHostServiceProcess(multiprocessing.Process):
    def __init__(self, project_name):
        super(AppHostServiceProcess, self).__init__(
            name='Kabaret_AppHostServiceProcess_'+project_name
        )
        self.project_name = project_name
        
    def run(self):
        print 'Looking for project server'
        project_server_address = find_service(
            'project:%s'%(self.project_name,), #exposure_port, response_port
        )
        if project_server_address is None:
            raise RuntimeError(
                'Sorry, the project %r was not found. Please contact an administrator'%(
                    self.project_name
                )
            )
        protocol, address_port = project_server_address.split('//', 1)
        protocol = protocol+'//'
        address, port = address_port.split(':')
        port = int(port)

        print 'Building AppHost Service'
        # Create the ProjectService and bind to random
        app_host_service = AppHostService(address, port, protocol)

        # Create and start an ExposeService sub process to serve the
        # AppHost's address
        import socket
        my_ip = socket.gethostbyname(socket.gethostname())
        services = {
            'apphost:%s@%s'%(self.project_name,my_ip): '%s://%s:%i'%(
                app_host_service.protocol,
                my_ip, 
                app_host_service.rpc_port
            ),
        }
        exposure_port = app_host_service.get_exposure_port()
        self.name_service = ExposeServices(
            name="Kabaret_apphost_NameServer:%s"%(self.project_name,),
            frequence=1, exposure_port=exposure_port,
            **services
        )
        self.name_service.daemon = True
        print 'Exposing AppHost Service on port', exposure_port
        self.name_service.start()

        print 'Starting Service'
        app_host_service.loop.start()



class AsyncRemoteScope(object):
    def __init__(self, client, scope):
        super(AsyncRemoteScope, self).__init__()
        self.client = client
        self.scope = scope

    def __call__(self, *args, **kwargs):
        return self.client.async(
            self.scope, *args, **kwargs
        )

    def __getattr__(self, name):
        return AsyncRemoteOperation(
            client=self.client,
            op=self.scope+'.'+name
        )
    
class AsyncRemoteOperation(object):
    def __init__(self, client, op):
        super(AsyncRemoteOperation, self).__init__()
        self.client = client
        self.op = op
          
    def __call__(self, *args, **kwargs):
        return self.client.async(
            self.op, *args, **kwargs
        )
    
class AppHostClient(AsyncClient):
    def __init__(self, adress, port):
        super(AppHostClient, self).__init__()
        self._app_event_dispatcher = EventDispatcher()
        self.connect(adress, port)
        self.subscribe('events')
        
    def __getattr__(self, name):
        return AsyncRemoteScope(self, name)

    def add_app_event_handler(self, handler, path, etype=None):
        if not self.ps_port:
            raise RuntimeError('subscribe must be called first')
        self._app_event_dispatcher.add_handler(
            handler, path, etype
        )
        
    def _receive_pub(self, msg_list):
        topic, data = msg_list
        pub_handler = {
            #TODO: This exists so that app can define 
            # some other PUB_TOPICS than Events.
            # Some of the app view would need to declare
            # the topic and set a pub_handler 
            PUB_TOPICS.EVENT: self._receive_app_event
        }.get(topic, None)
        if pub_handler is not None:
            pub_handler(self._serializer.loads(data))
            
    def _receive_app_event(self, data):
        event = Event.from_dict(data)
        self._app_event_dispatcher.dispatch(event)                    
            
    def emit_app_event(self, event):
        self.one_way('publish', PUB_TOPICS.EVENT, event.to_dict())





def _test():
    import sys
    args = list(sys.argv[1:])
    if len(args) < 2:
        print 'usage: %s <S/C> <project> [store_path]'%(sys.argv[0],)
        raise SystemExit
    
    port = 1000
    
    if args.pop(0) == 's':
        project_name = args.pop(0)
        store_path = args and args.pop(0) or 'X:/TEST_STORE' 
        service = AppHostService(store_path, project_name, port)
        service.start()
    else:
        import traceback
        client = AppHostClient('192.168.1.108', port)
        alive = True
        context = {'c':client}
        result = None
        while alive:
            i = raw_input('Client>')
            if not i.strip():
                client.tick()
                continue
            if i.strip() == 'q':
                alive = False
            else:
                try:
                    result = eval(i, context)
                    print ' ', i
                    context['_'] = result
                except SyntaxError:
                    exec(i, context)
                except:
                    traceback.print_exc()
                else:
                    if result is not None:
                        print '>', result


if __name__ == '__main__':
    _test()
    