'''
    Copyright (c) Supamonks Studio and individual contributors.
    All rights reserved.

    This file is part of kabaret, a python Digital Creation Framework.

    Kabaret is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.
        
    Redistributions in binary form must reproduce the above copyright 
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.
    
    Kabaret is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.
    
    You should have received a copy of the GNU Lesser General Public License
    along with kabaret.  If not, see <http://www.gnu.org/licenses/>

--

    The kabaret.flow.params.case module.
    Defines the CaseParam class.
    A Node using some CaseParam is able to represent different scenarios 
    saved in some kabaret.flow.case.CaseData
    
'''

from .param import Param

class CaseParam(Param):
    '''
    A CaseParam is able to set the ParamValue accordingly to
    the data in the node case.
    '''
    def __init__(self, default=None):
        super(CaseParam, self).__init__()
        self.default = default
        
    def apply_case(self, node):
        '''
        Called when the node owning this Param loads a
        case.
        Sets the ParamValue accordingly to the data in 
        the node's case.
        '''
        pv = self.get_value(node)
        try:
            pv.set(node._case[self.name])
        except KeyError:
            if self.default is None:
                raise
            value = callable(self.default) and self.default() or self.default
            pv.set(value)
            node._case[self.name] = value
                
    def create_case(self):
        '''
        Called when the node owning this Param is building
        a case.
        Return the ParmaValue's default data.
        '''
        default = callable(self.default) and self.default() or self.default
        return {self.name: default}
