'''
    Copyright (c) Supamonks Studio and individual contributors.
    All rights reserved.

    This file is part of kabaret, a python Digital Creation Framework.

    Kabaret is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.
        
    Redistributions in binary form must reproduce the above copyright 
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.
    
    Kabaret is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.
    
    You should have received a copy of the GNU Lesser General Public License
    along with kabaret.  If not, see <http://www.gnu.org/licenses/>

--

    The kabaret.flow.relations.many module.
    Defines the Many class.
    This relation tells that one related Node exists for 
    each sub-case in the parent node case.
        
    For example, a Film contains several Shots.
    You don't want to write a specific Node class for each film, you
    want to write a Film Node that contains several (one per case)
    Shots Nodes.

    The related nodes are lazily created.
    
'''

from ._base import Relation


class Many(Relation):
    '''
    The Many Relation ties several nodes to its owner node.
    '''
    def __init__(self, node_type, default_case_id=None):
        '''
        The default_case_id is used when creating a case with
        the create_case() method.
        '''
        super(Many, self).__init__(node_type)
        self.default_case_id = default_case_id

    def _create_related_node(self, parent):
        '''
        The Many Relation does not create node directly,
        it must use a item name to specify the case id 
        of one of the related nodes:
            related_node = my_node.many_relation[case_id]
            
        So this method raises an AttributeError.
        '''
        raise AttributeError(
            'This must be done by the _CaseProducer'
        )
        
    def __get__(self, node, node_type):
        '''
        Returns the Many relation when called from the
        class.
        Returns a _CaseProducer when called from a Many
        relation instance.
        (See the _CaseProducer class.)
        '''
        if node is None:
            return self
        return _CaseProducer(self, node)

    def create_case(self):
        '''
        Creates default case for this relation.
        The returned case will contain a case created
        by the related node's type create_case() method
        stored under the name given in the constructor's
        default_case_id argument.
        '''
        case_id = self.default_id or self.name.title()
        cases = {
            case_id:self.node_type.create_case(),
        }
        return {self.name: cases}

class _CaseProducer(object):
    '''
    A _CaseProducer is used by the Many Relation to specify
    the case id of the related node to retrieve:
        related_node = my_node.many_relation[case_id]
    
    The _CaseProducer can list the available case id with
    its get_case_ids() method:
        case_ids = my_node.many_relation.get_case_ids()
        
    '''
    def __init__(self, cases_relation, parent_node):
        super(_CaseProducer, self).__init__()
        self.parent_node = parent_node
        self.cases_relation = cases_relation
    
    def get_case_ids(self):
        '''
        Returns the list of case id available for this relation.
        '''
        return list(self.parent_node._case[self.cases_relation.name].iterkeys())
    
    def __getitem__(self, name):
        '''
        Returns the node in this relation having the case id 'name'.
        '''
        key = (self.cases_relation.name, name)
        try:
            return self.parent_node._many_related[key]
        except KeyError:
            node = self.cases_relation.node_type(
                self.parent_node, name
            )
            self.parent_node._many_related[key] = node
            self.cases_relation._configure_node(node)
            case = self.parent_node._case[self.cases_relation.name][name]
            node.set_case(case, self.cases_relation.name)            
            return node
