'''
    Copyright (c) Supamonks Studio and individual contributors.
    All rights reserved.

    This file is part of kabaret, a python Digital Creation Framework.

    Kabaret is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.
        
    Redistributions in binary form must reproduce the above copyright 
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.
    
    Kabaret is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.
    
    You should have received a copy of the GNU Lesser General Public License
    along with kabaret.  If not, see <http://www.gnu.org/licenses/>

--

    The kabaret.flow.case package.
    Defines the base class CaseData for case data handling.
    
'''

class CaseData(object):
    '''
    The CaseData class is used by the node instances
    to access the CaseParam data of a case.
    
    The Flow holding your nodes needs a CaseData in order
    to initialize its root node.
    The root children nodes will use the CaseData from
    the root and so on for sub-children.
    This is how the Case is dispatched to the node tree.
    
    This implementation stores everything in an
    internal dict.
    You will need to subclass it and implement some
    persistence/IO in order to be useful.
    While doing so, you will want to delay the IO to 
    the last moment so that you don't load everything
    at once.
    
    '''
    
    def __init__(self, parent=None, name=None):
        super(CaseData, self).__init__()
        self.parent = parent
        self.name = name
        self.data = {}
    
    def add_group(self, name):
        ret = self.__class__(self, name)
        self.data[name] = ret
        return ret

    def __setitem__(self, name, value):
        self.data[name] = value
        
    def __getitem__(self, name):
        try:
            return self.data[name]
        except KeyError:
            raise KeyError(
                'CaseData not found: %s in %r'%( 
                    self.path(name), self
                )
            )
    
    def path(self, extra_child_name=None):
        path = []
        if self.parent is not None:
            path.append(self.parent.path())
        if self.name is not None:
            path.append(self.name)
        if extra_child_name is not None:
            path.append(extra_child_name)
        path = [ i for i in path if i]
        return '.'.join(path)
        
    def __repr__(self):
        return '<CaseData: %s %r>'%(self.path(), self.data)
    
    def pformat(self, indent=0):
        ret = ['Group: %s'%(self.name,)]
        for k, v in self.data.iteritems():
            if isinstance(v, self.__class__):
                ret.append(('  '*indent)+v.pformat(indent+1))
            else:
                ret.append(('  '*indent)+'%s = %r'%(self.path(k), v))
        return '\n'.join(ret)
    
    def iterkeys(self):
        return self.data.iterkeys()
