'''
    Copyright (c) Supamonks Studio and individual contributors.
    All rights reserved.

    This file is part of kabaret, a python Digital Creation Framework.

    Kabaret is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.
        
    Redistributions in binary form must reproduce the above copyright 
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.
    
    Kabaret is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.
    
    You should have received a copy of the GNU Lesser General Public License
    along with kabaret.  If not, see <http://www.gnu.org/licenses/>

--

    The kabaret.flow.nodes.node module.
    Defines the Node class, base for all flow nodes.
    
'''

from ..params.param import Param
from ..params.case import CaseParam
from ..relations._base import Relation
from ..relations.child import Child


class NodeType(type):
    '''
    The NodeType is the Node meta class.
    
    It sets the name of all Relation and Param defined in the
    class definition.
    
    It collects Relation and Param in the base class(es) and build
    the new Node class' _child_relations, _relations, and _params lists.
    
    '''
    
    def __new__(cls, class_name, bases, class_dict):
        is_base_class = bases == (object,)
        
        if not is_base_class:
            # Check name clashes:
            if '_params' in class_dict:
                raise ValueError('"_params" is a reserved name in a Node subclass.')
            if '_child_relations' in class_dict:
                raise ValueError('"_child_relations" is a reserved name in a Node subclass.')
            if '_relations' in class_dict:
                raise ValueError('"_relations" is a reserved name in a Node subclass.')
        
            # Collect base classes infos:
            _params = []
            _child_relations = []
            _relations = []
            for base in bases:
                if issubclass(base, Node):
                    _params.extend(base._params)
                    _child_relations.extend(base._child_relations)
                    _relations.extend(base._relations)
        
            # Gather and Configure all descriptors:
            for n, o in class_dict.iteritems():
                if isinstance(o, Param):
                    o.name = n
                    _params.append(o)
                elif isinstance(o, Child):
                    o.name = n
                    _child_relations.append(o)
                elif isinstance(o, Relation):
                    o.name = n
                    _relations.append(o)
            
            # Extend class definition:
            class_dict['_params'] = _params
            class_dict['_child_relations'] = _child_relations
            class_dict['_relations'] = _relations
            
            if 0:
                print 5*'###########\n'
                print 'SUBCLASS', class_name
                print 'SUPERS:', bases
                print '_params:', [ p.name for p in _params ]
                print '_child_relations:', [ n.name for n in _child_relations ]
                print '_relations:', [ r.name for r in _relations ]
            
        # Instanciate the class:
        return  super(NodeType, cls).__new__(cls, class_name, bases, class_dict)

class Node(object):
    '''
    The Node holds some Param and is tied to other nodes by some Relation.
    
    The connections between nodes' Param form a directed multigraph.
    The related nodes form a tree.
    
    A Node belongs to a Flow.
    The Flow provides functionalities specific to the purpose and context of
    the node graph.
    
    The ComputedParam will call the node's compute() method to update
    their data.
    The node's compute() method should call the ComputedParamValue's set() 
    method.

    The CaseParam will have their value read from the node's case.

    When a ParamValue gets dirty (some source value has changed), the node's 
    param_touched() is called.
    The node's param_touched is responsible of the propagation of the touch
    to its other Params.
    
    For convenience and rare dependency cases, the node also receive a call
    to param_cleaned when a param is set.
    '''
    
    __metaclass__ = NodeType
    
    _params = []
    _relations = []
    _child_relations = []
    
    @classmethod
    def create_case(cls, case_type={}):
        '''
        Return a new case with default values
        for this node class and all related nodes.
        
        The case_type must be a callable w/o argument
        that initialize an empty case.
        (The Flow you use should be able to provide it)
        
        '''
        ret = case_type()
        for param in cls.iterparams():
            ret.update(param.create_case())
            
        for relation in cls._relations:
            ret.update(relation.create_case())
        
        for child_descriptor in cls._child_relations:
            ret.update(child_descriptor.create_case())
            
        return ret
    
    def __init__(self, parent, node_id=None):
        '''
        Instantiates a new node.
        You should not need to create nodes yourself since
        the Flow.init_root() will create the root node and
        all related nodes will be accessible thru their owner
        node.
        
        All Child nodes are created when creating a Node.
        
        '''
        
        super(Node, self).__init__()
        self._parent = parent
        self.node_id = node_id or self.__class__.__name__
        self._flow = None
        self._case = None
        self._parent_relation_name = None
        
        self._one_related = {}    # One relation name: related node instance
        self._many_related = {}   # (Many relation name, case id) : related node instance
        self._child_related = {}   # Child relation name: related node instance
        self._param_values = {}      # Param name: ParamValue
        
        self._init_child_nodes()
    
    def _init_child_nodes(self):
        '''
        Called by the node constructor to initialize all 
        Child nodes.
        '''
        # iterate over all ChildNode relation and trigger
        # them to ensure child nodes are created and configured
        for child_node_relation in self._child_relations:
            getattr(self, child_node_relation.name)
        
    def set_case(self, case, parent_relation_name=None):
        '''
        Sets the case used by this node.
        Each Param will get an apply_case() call.
        Each Child node will get a set_case() call with a
        corresponding sub-case if at least one o
        '''
        self._case = case
        self._parent_relation_name = parent_relation_name
        for param in self.iterparams():
            param.apply_case(self)
        
        for child_node in self._child_related.values():
            if 0:
                for param in child_node.iterparams():
                    if isinstance(param, CaseParam):
                        # set case only if the node has some CaseParam
                        child_node.set_case(self._case[child_node.node_id])
                        break
            else:
                # we always set_case since a dumb grouping
                # node might be on the way to a CaseParam
                child_node.set_case(self._case[child_node.node_id])

    def has_param(self, param_name):
        '''
        Returns True if the node has a Param with name
        'param_name'
        '''
        return param_name in self._param_values

    def get_param(self, param_name):
        '''
        Returns the Param named 'param_name'
        (Not the ParamValue, you must use get_param_value
        or attribute access for that.)
        '''
        return getattr(self.__class__, param_name)
    
    def get_param_value(self, param_name):
        '''
        my_node.get_param_value('my_param') <=> my_node.my_param
        
        If you need to access the Param instead of the ParamValue,
        you must call get_param('my_param').
        '''
        return getattr(self.__class__, param_name).get_value(self)
    
    @classmethod
    def iterparams(cls):
        '''
        Returns an iterator on the list of Param in this Node class.
        '''
        return iter(cls._params)

    def iterchildren(self):
        '''
        Returns an iterator of (relation_name, child_node) for
        all child related node in this node.
        '''
        return self._child_related.iteritems()
            
    @classmethod
    def iterrelations(cls):
        '''
        Returns an iterator on each relation (not including
        Child relations) of this Node class.
        '''
        return iter(cls._relations)

    def flow(self):
        '''
        Returns the flow managing this node.
        If this node is not the root node, its parent
        flow() method is used.
        
        If this node is not a flow root and not related
        to a node knowing its flow, None is returned.
        (Which is not likely to happen)
        '''
        if self._flow is not None:
            return self._flow
        if self._parent is not None:
            return self._parent.flow()
        return None

    def parent(self):
        '''
        Returns this node's parent node.
        
        The node parent is the one holding the relation
        that ties both node together.
        '''
        return self._parent

    def uid(self):
        '''
        Returns a unique identifier for this node in its flow.
        
        This uid can be used on the node's flow's get() method
        to later retrieve this node.
        
        The uid is a tuple of string.
        Popping the tail of this tuple give successive parents
        uid.
        '''
        nid = self.node_id
        if self._parent_relation_name:
            nid = ':'.join((self._parent_relation_name, nid))
        uid = (nid,)
        if self._parent is not None:
            uid = self._parent.uid() + uid
        return uid
    
    def get(self, relative_uid):
        '''
        Returns a node related to this one using
        the given relative_uid.
        '''
        node = self
        relative_uid = list(relative_uid)
        while relative_uid:
            next_id = relative_uid.pop(0)
            if next_id.startswith('.'):
                return self.get_param_value(next_id[1:])
            elif ':' in next_id:
                # it is a relation:
                relation, case_id = next_id.split(':')
                node = node[relation][case_id]
            else:
                # it is a ChildNode:
                node = node[next_id]
        return node
                
    def path(self):
        '''
        Returns a file like path unique in the flow
        that designate this node.
        This is only for display purpose and cannot
        be used to later retrieve this node (use uid()
        for that).
        '''
        return '/'.join(self.uid())
    
    def param_touched(self, param_name):
        '''
        Called by a Param of this node when it got touched (goes
        to dirty because one of its source changed).
        
        Subclasses should re-implement this to propagate the 
        touch() to all ComputedParam which value depend on
        this one.
        '''
        if 0:
            print 'Touched', self.get_param_value(param_name).path()
            print '  UPS:', [ pv.path() for pv in self.get_param_value(param_name).upstreams ]
            print '  DOWNS:', [ pv.path() for pv in self.get_param_value(param_name).downstreams ]

    
    def compute(self, param_name):
        '''
        Computes and set the value of the node's Param named 'param_name'.
        If this method leaves w/o setting the param value, a
        kabaret.flow.params.computed.ComputError will be raised.
        
        Subclasses must implement this method if they hold some
        ComputedParam.
        '''
        print 'Compute', self.get_param_value(param_name).path()
        print '  UPS:', [ pv.path() for pv in self.get_param_value(param_name).upstreams ]
        print '  DOWNS:', [ pv.path() for pv in self.get_param_value(param_name).downstreams ]
        raise NotImplementedError

    def param_cleaned(self, param_name):
        '''
        Called by a Param of this node when it got set.
        
        Subclasses should re-implement this if needed.
        (Which is quite uncommon.)
        '''
        if 0:
            print 'Cleaned', self.get_param_value(param_name).path()
            print '  UPS:', [ pv.path() for pv in self.get_param_value(param_name).upstreams ]
            print '  DOWNS:', [ pv.path() for pv in self.get_param_value(param_name).downstreams ]
