'''
    Copyright (c) Supamonks Studio and individual contributors.
    All rights reserved.

    This file is part of kabaret, a python Digital Creation Framework.

    Kabaret is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.
        
    Redistributions in binary form must reproduce the above copyright 
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.
    
    Kabaret is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.
    
    You should have received a copy of the GNU Lesser General Public License
    along with kabaret.  If not, see <http://www.gnu.org/licenses/>

--

    The kabaret.flow.params.computed module.
    Defines the ComputedParam and ComputedParamValue.
    A Node using ComputedParam is able to provide dynamic value. 
    This is how the flow provides intelligence.
    
'''
from .param import ParamValue, Param

class ComputeError(AttributeError):
    def __init__(self, node, param_name, message):
        self.node = node
        self.param_name = param_name
        self.message = message
        super(ComputeError, self).__init__(
            'Error computing %r in %r: %s'%(
                param_name, node.path(), message
            )
        )

class ComputedParamValue(ParamValue):
    '''
    The ComputedParamValue call its node's compute_param() when
    its get() method is call while it is dirty.
    
    The node compute_param must then call set() on this ParamValue
    or a ComputeError will be raised.
    Any Exception raise while in compute_param will be wrapped in
    a ComputeError, stored in the 'error' attribute and raised.
    '''
    def __init__(self, param_name, node):
        super(ComputedParamValue, self).__init__(param_name, node)
        self.error = None
        
    def get(self):
        if self.error:
            raise self.error

        if self._dirty:
            try:
                self.node.compute(self.param_name)
            except Exception, err:
                self.error = ComputeError(self.node, self.param_name, str(err))
                raise self.error
            if self._dirty:
                self.error = ComputeError(
                    self.node, self.param_name, 
                    "compute() did not compute me! (i\'m still dirty)"
                )
                raise self.error
        
        return self.value

class ComputedParam(Param):
    _VALUE_CLASS = ComputedParamValue
