from datetime import datetime, timedelta
import tarfile
from plot.parser import parse_lines, LogParser


def parse(pr, infile, db, tablename):
    def ms_since_epoch(logstart, interval):
        # helper to create a generator for adding timestamps to 
        # parsed loglines
        # workaround missing nonlocal to implement closure
        nonlocal = {
            'logstart' : logstart,
            'interval': int(interval)
            }

        def gen(**kw_args):
            if 'ts' in kw_args:
                ts = datetime.strptime(
                    kw_args['ts'], '%H:%M:%S').time()
                low = (nonlocal['logstart'] - 
                    timedelta(seconds=1)).time()
                high = (nonlocal['logstart'] + 
                    timedelta(seconds=1)).time()
                if not (low <= ts <= high):
                    nonlocal['logstart'] += timedelta(
                        seconds=nonlocal['interval'])
            return nonlocal['logstart'].strftime('%s%f')[:13]
        return gen

    # read logstart and interval from file
    match = pr.compile_regex('%{COUNTER_START}').match(
        infile.readline()).groupdict()

    callbacks = {'ms_since_epoch': ms_since_epoch(
        datetime.strptime(match['logstart'], '%Y%m%d %H%M%S'),
        match['interval'])
    }

    # parse the data from the logfiles
    top = LogParser(pr.compile_regex('%{LINUX_TOP}'), callbacks, db, tablename)
    clock = LogParser(pr.compile_regex('%{LINUX_TOP_CLOCK}'), callbacks=callbacks)
    discard = LogParser(pr.compile_regex('%{LINUX_TOP_DISCARD}'))
    parse_lines([top, clock, discard], infile)


def populate(pr, filename, db, testrun, tablename):
    with tarfile.open(filename, mode="r:gz") as tar:
        infile = tar.extractfile('./top.txt.%s' % testrun)
        parse(pr, infile, db, tablename)


def query(db, tablename, arguments=[]):
    data = []
    c = db.cursor()
    c.execute('select command, sum(cpu) as cpu from %s ' % tablename
        + 'group by command '
        + 'order by cpu desc')
    commands = zip(*c.fetchmany(size=9))[0]
    # timestamp & others
    c.execute('select ms_since_epoch, '
        + 'sum(cpu) as cpu from %s where ' % tablename
        + 'command NOT IN ("%s") ' % '", "'.join(commands)
        + 'group by ms_since_epoch order by ms_since_epoch')
    timestamp, others = zip(*c.fetchall())
    data.append(['ms_since_epoch'] + list(timestamp))
    # insert missing zero values
    for ts in timestamp:
        for command in commands:
            c.execute('insert into %s(ms_since_epoch, command, ' % tablename
                + 'cpu) values (?, ?, ?)', (ts, command, 0))
    # CPU utilization for commands
    for command in commands:
        c.execute('select sum(cpu) as cpu from %s where command=?' % tablename
            + 'group by ms_since_epoch order by ms_since_epoch', (command,))
        data.append([command] + list(zip(*c.fetchall())[0]))
    data.append(['all other processes'] + list(others))
    c.close()

    return zip(*data)  # transpose matrix

