"""
Support for exporting database to KeePassX XML format.
"""
from __future__ import absolute_import
from datetime import datetime
from xml.etree import ElementTree as ET
from xml.dom import minidom

from keepassdb import const

class XmlExporter(object):
    """
    Class for exporting database to KeePassX XML format.
    
    :ivar include_comment: Whether to include a 'generated-by' comment in the header.
    :ivar prettyprint: Whether to generate pretty-printed XML (indent, etc.).
    """
    include_comment = False
    prettyprint = True
    
    def __init__(self, include_comment=False, prettyprint=True):
        self.include_comment = include_comment
        self.prettyprint = prettyprint
    
    def export(self, db):
        """
        Export the dbnode to KeePassX XML format.
        
        :param db: The dbnode to export.
        :type db: :class:`keepassdb.db.Database` 
        """
        dbnode = ET.Element('database')
    
        if self.include_comment:
            now = datetime.now()
            filepath = db.filepath
            if filepath:
                comment = ET.Comment('Generated by keepassdb from {0} on {1}'.format(filepath, now.strftime("%c")))
            else:
                comment = ET.Comment('Generated by keepassdb on {0}'.format(now.strftime("%c")))
            dbnode.append(comment)
        
        def _date(dt):
            if dt == const.NEVER:
                return 'Never'
            else:
                # 2012-12-20T20:56:56
                return dt.strftime('%Y-%m-%dT%H:%M:%S')
            
        def group_to_xml(group, node):
            gnode = ET.SubElement(node, 'group')
            title = ET.SubElement(gnode, 'title')
            title.text = group.title
            icon = ET.SubElement(gnode, 'icon')
            icon.text = str(group.icon)
            
            for subgroup in group.children:
                group_to_xml(subgroup, gnode)
            
            for entry in group.entries:
                if entry.title == 'Meta-Info' and entry.username == 'SYSTEM':
                    continue
                enode = ET.SubElement(gnode, 'entry')
                ET.SubElement(enode, 'title').text = entry.title
                ET.SubElement(enode, 'username').text = entry.username
                ET.SubElement(enode, 'password').text = entry.password
                ET.SubElement(enode, 'url').text = entry.url
                ET.SubElement(enode, 'comment').text = entry.notes
                ET.SubElement(enode, 'icon').text = str(entry.icon)
                ET.SubElement(enode, 'creation').text = _date(entry.created)
                ET.SubElement(enode, 'lastaccess').text = _date(entry.accessed)
                ET.SubElement(enode, 'lastmod').text = _date(entry.modified)
                ET.SubElement(enode, 'expire').text = _date(entry.expires)
            return gnode
        
        for group in db.root.children:
            dbnode.append(group_to_xml(group, dbnode))
            
        xmlstr = ET.tostring(dbnode)
        if self.prettyprint:
            reparsed = minidom.parseString(xmlstr)
            xmlstr = reparsed.toprettyxml(indent=" ")
        
        return xmlstr