"""
Implements the core :class:`Graph <kenji.graph.Graph>` interface.
"""

from sqlite3 import Connection
from contextlib import closing
from kenji.query import Query
import kenji.sql as sql

__all__ = ['Graph']


class Graph(object):
    """
    The main class that encompasses storage and deletion of the
    relations between nodes, and gives access to querying.
    """

    def __init__(self, uri=':memory:', graphs=[], unsafe=False, mapping=None):
        """
        Create a new ``Graph`` instance.

        :param uri: The URI of the SQLite db
        :param graphs: List of relations to create
        :param unsafe: Whether to turn off ``PRAGMA synchronous``
        :param mapping: A dictionary of string to iterables which
        dictates the mapping that should be respected when we
        do relation searches.
        """
        self.uri = uri
        self.db = Connection(uri)
        self.closed = False
        cursor = self.db.cursor()
        with closing(cursor):
            for graph in graphs:
                cursor.execute(sql.CREATE_TABLE % (graph))
                for index in sql.INDEXES:
                    cursor.execute(index % (graph))
            if unsafe:
                cursor.execute('PRAGMA synchronous = OFF;')
        self.mapping = {} if mapping is None else mapping

    def store(self, edge):
        """
        Store a relation to the SQLite backend. The relation must
        already be specified during the creation of the graph.

        :param edge: An edge to store
        """
        relation = edge.relation
        cursor = self.db.execute(
            *sql.write_relation(edge.src, relation, edge.dst)
        )
        with closing(cursor):
            self.db.commit()

    def delete(self, edge):
        """
        Delete (possibly multiple) edges from the database.

        :param edge: The edge(s) to delete.
        """
        relation = edge.relation
        cursor = self.db.execute(
            *sql.delete_relation(edge.src, relation, edge.dst)
        )
        with closing(cursor):
            self.db.commit()

    def close(self):
        """
        Close the database and the SQLite backend.
        """
        if not self.closed:
            self.db.close()
            self.closed = True

    __del__ = close

    def select(self, *args, **kwargs):
        """
        Returns a new :class:`Query <kenji.query.Query>` object
        that you can iterate or perform filters on. Arguments
        are forwarded to the ``__call__`` method of the query
        object.
        """
        return Query(self.db)(*args, **kwargs)

    def exists(self, *args, **kwargs):
        """
        Checks if an edge exists, a proxy for the :meth:``Query.exists``
        method.
        """
        return Query(self.db).exists(*args, **kwargs)

    def relation_between(self, type, edge):
        """
        Query the relation between two different nodes,
        with the restriction on which tables to query
        depending on the type of the nodes as defined
        in the `mapping` argument to the constructor.

        :param type: The type restriction.
        :param from_node: A source node.
        :param to_node: A destination node.
        """
        from_node, to_node = edge.src, edge.dst
        tables = self.mapping[type]
        cursor = self.db.cursor()
        with closing(cursor):
            relations = []
            for table in tables:
                cursor.execute(*sql.select_one_relation(from_node, table, to_node))
                if tuple(cursor):
                    relations.append(table)
            return relations
