"""
Implements a :class:`Query <kenji.query.Query>` object that
allows querying of the relations and edges.
"""

from contextlib import closing
from functools import wraps
import kenji.sql as sql


class V(object):
    """
    Represents an edge, but is named ``V`` because it
    seems more intuitive and because it's inspired by
    Gremlin queries.
    """

    def __init__(self, value=None):
        """
        Creates a new ``V`` object.

        :param value: Defaults to none, a source node.
        """
        self.src = value
        self.relation = None
        self.dst = None

    def __call__(self, value):
        """
        Assign the destination node to a given value.

        :param value: The destination node.
        """
        self.dst = value
        return self

    def __getattr__(self, attribute):
        values = self.__dict__
        if attribute in values:
            return values[attribute]
        self.relation = attribute
        return self


class Query(object):
    """
    The main querying interface.
    """

    def __init__(self, db):
        """
        Creates a new querying instance tied to the given
        connection object.

        :param db: A SQLite Connection object.
        """
        self.db = db
        self.sql = []
        self.params = []

    def __call__(self, query):
        """
        Make a query that pulls either the source or
        destination node depending on the given node
        query.

        :param query: The node query.
        """
        relation = query.relation
        src = query.src
        statement, params = (
            sql.inverse_relationship(query.dst, relation) if src is None else
            sql.forwards_relationship(src, relation)
        )
        self.sql.append(statement)
        for parameter in params:
            self.params.append(parameter)
        return self

    def exists(self, edge):
        """
        Checks if an edge exists.

        :param edge: The edge to check.
        """
        cursor = self.db.execute(
            *sql.select_one_relation(edge.src, edge.relation, edge.dst)
        )
        with closing(cursor):
            return bool(tuple(cursor))

    @property
    def union(self):
        """
        Injects an SQL ``UNION`` keyword at the end of
        the internal SQL string and returns the query
        object.
        """
        self.sql.append('UNION')
        return self

    @property
    def intersection(self):
        """
        Injects an SQL ``INTERSECT`` keyword at the end
        of the internal SQL string and returns the query
        object.
        """
        self.sql.append('INTERSECT')
        return self

    @property
    def difference(self):
        """
        Injects an SQL ``EXCEPT`` keyword at the end
        of the internal SQL string and returns the query
        object.
        """
        self.sql.append('EXCEPT')
        return self

    def __iter__(self):
        sql = '\n'.join(self.sql)
        cursor = self.db.execute(sql, self.params)
        with closing(cursor):
            for item in cursor:
                yield item[0]

    def limit(self, count):
        """
        Adds an SQL ``LIMIT`` constraint to the end
        of the internal SQL query, and returns the
        query object.

        :param count: The integer limit.
        """
        self.sql.append('LIMIT %d' % (count))
        return self

    @property
    def first(self):
        """
        Inserts a ``LIMIT 1`` constaint at the end of
        the internal SQL query and returns the next
        node yielded by performing the query.
        """
        return next(iter(self.limit(1)))

    @property
    def count(self):
        """
        Count the number of nodes returned in a query.
        """
        counter = 0
        for _ in self:
            counter += 1
        return counter
