"""
Contains commonly used SQL expressions that are meant to
be commonly used and modular in nature.
"""

CREATE_TABLE = '''\
CREATE TABLE IF NOT EXISTS %s
(
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    to_node UNSIGNED INTEGER,
    from_node UNSIGNED INTEGER
);\
'''

INDEXES = [
    'CREATE INDEX IF NOT EXISTS forwards_index ON %s ( to_node );',
    'CREATE INDEX IF NOT EXISTS backwards_index ON %s ( from_node );'
]


def write_relation(from_node, relation, to_node):
    """
    Write a relation into the a table named after the
    relation.

    :param from_node: An integer source node
    :param to_node: An integer destination node
    """
    statement = "INSERT INTO %s (from_node, to_node) VALUES (?,?);"
    return statement % (relation), (from_node, to_node)


def delete_relation(from_node, relation, to_node):
    """
    Delete a relation from a node to another node, from
    a given table that must already exist.

    :param from_node: May be None, the source node.
    :param relation: The relation.
    :param to_node: A destination node.
    """
    statement = 'DELETE FROM %s' % (relation)
    queries = []
    params = []

    if from_node is not None:
        queries.append('from_node = ?')
        params.append(from_node)

    if to_node is not None:
        queries.append('to_node = ?')
        params.append(to_node)

    if not queries:
        return statement, params
    statement = '%s WHERE %s' % (statement, ' AND '.join(queries))
    return statement, params


# Querying

def forwards_relationship(from_node, relation):
    """
    Select destination nodes given a relation and
    a source node.

    :param from_node: An integer source node
    :param relation: The relation to query
    """
    statement = "SELECT to_node FROM %s WHERE from_node = ?"
    return statement % (relation), (from_node,)


def inverse_relationship(to_node, relation):
    """
    Select source nodes given a destinatio node
    and a relation.

    :param to_node: An integer destination node
    :param relation: The relation to query
    """
    statement = "SELECT from_node FROM %s WHERE to_node = ?"
    return statement % (relation), (to_node,)


def select_one_relation(from_node, relation, to_node):
    """
    Try to select one ``id`` column from a given edge.
    All arguments must be specified.

    :param from_node: The source node.
    :param relation: The relation.
    :param to_node: A destination node.
    """
    statement = (
        'SELECT id FROM %s'
        ' WHERE from_node = ? AND to_node = ? LIMIT 1;'
    )
    return statement % (relation), (from_node, to_node)
