"""
Implements a :class:`Query <kenji.query.Query>` object that
allows querying of the relations and edges.
"""

from contextlib import closing
from functools import wraps
import kenji.sql as sql

__all__ = ['V', 'Query']


class V(object):
    """
    Represents an edge, but is named ``V`` because it
    seems more intuitive and because it's inspired by
    Gremlin queries.
    """

    def __init__(self, src=None, relation=None, dst=None):
        """
        Creates a new ``V`` object.

        :param src: Defaults to None, a source node.
        :param relation: The relation, defaults to None.
        :param dst: A destination node, defaults to None.
        """
        self.src = src
        self.relation = relation
        self.dst = dst

    def __ge__(self, other):
        """
        Create a new edge which source node is the
        source node of the edge on the left, and the
        destination node is the source node of the one
        on the right. Example:

            >>> z = V(1) >= V(2)
            >>> z.src == 1 and z.dst == 2
            True

        :param other: The other edge.
        """
        v = V(self.src)
        v.dst = other.src
        return v

    def __le__(self, other):
        """
        Pragmatically creates another edge object-
        best explained by example:

            >>> z = V(2) <= V(1)
            >>> z.dst == 2 and z.src == 1
            True

        :param other: The other edge.
        """
        v = V(other.src)
        v.dst = self.src
        return v

    def __repr__(self):
        return '(%s)-[%s]->(%s)' % (
            '*' if self.src is None else self.src,
            '*' if not self.relation else ':%s' % (self.relation),
            '*' if self.dst is None else self.dst
        )

    def __call__(self, value=None):
        """
        Assign the destination node to a given value.

        :param value: The destination node.
        """
        self.dst = value
        return self

    def __getattr__(self, attribute):
        values = self.__dict__
        if attribute in values:
            return values[attribute]
        self.relation = attribute
        return self


class Query(object):
    """
    The main querying interface.
    """

    def __init__(self, db):
        """
        Creates a new querying instance tied to the given
        connection object.

        :param db: A SQLite Connection object.
        """
        self.db = db
        self.query = []
        self.params = []

    def __call__(self, edge):
        """
        Make a query that pulls either the source or
        destination node depending on the given node
        query. The relation must be specified on the
        node query.

        :param edge: An edge query.
        """
        relation = edge.relation
        src, dst = edge.src, edge.dst
        statement, params = (
            sql.inverse_relationship(dst, relation) if src is None else
            sql.forwards_relationship(src, relation)
        )
        self.query.append(statement)
        self.params.extend(params)
        return self

    @property
    def union(self):
        """
        Injects an SQL ``UNION`` keyword at the end of
        the internal SQL string and returns the query
        object.
        """
        self.query.append('UNION')
        return self

    @property
    def intersection(self):
        """
        Injects an SQL ``INTERSECT`` keyword at the end
        of the internal SQL string and returns the query
        object.
        """
        self.query.append('INTERSECT')
        return self

    @property
    def difference(self):
        """
        Injects an SQL ``EXCEPT`` keyword at the end
        of the internal SQL string and returns the query
        object.
        """
        self.query.append('EXCEPT')
        return self

    def __iter__(self):
        statement = '\n'.join(self.query)
        with closing(self.db.cursor()) as cursor:
            cursor.execute(statement, self.params)
            for item in cursor:
                yield item[0]

    def limit(self, count):
        """
        Adds an SQL ``LIMIT`` constraint to the end
        of the internal SQL query, and returns the
        query object.

        :param count: The integer limit.
        """
        self.query.append('LIMIT %d' % (count))
        return self

    @property
    def first(self):
        """
        Inserts a ``LIMIT 1`` constaint at the end of
        the internal SQL query and returns the next
        node yielded by performing the query.
        """
        try:
            return next(iter(self.limit(1)))
        except StopIteration:
            return

    @property
    def count(self):
        """
        Count the number of nodes returned in a query.
        """
        return sum(1 for row in self)
