"""
Contains commonly used SQL expressions that are meant to
be commonly used and modular in nature.
"""

__all__ = []


CREATE_TABLE = '''\
CREATE TABLE IF NOT EXISTS %s
(
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    src UNSIGNED INTEGER,
    dst UNSIGNED INTEGER
);\
'''

INDEXES = [
    'CREATE INDEX IF NOT EXISTS src_index ON %s ( src );',
    'CREATE INDEX IF NOT EXISTS dst_index ON %s ( dst );'
]


def write_relation(src, relation, dst):
    """
    Write a relation into the a table named after the
    relation.

    :param src: An integer source node
    :param relation: A relation.
    :param dst: An integer destination node
    """
    statement = 'INSERT INTO %s (src, dst) VALUES (?,?);'
    return statement % (relation), (src, dst)


def delete_relation(src, relation, dst):
    """
    Delete a relation from a node to another node, from
    a given table that must already exist.

    :param src: May be None, the source node.
    :param relation: The relation.
    :param dst: A destination node.
    """
    statement = 'DELETE FROM %s' % (relation)
    queries = []
    params = []

    if src is not None:
        queries.append('src = ?')
        params.append(src)

    if dst is not None:
        queries.append('dst = ?')
        params.append(dst)

    if not queries:
        return statement, params
    statement = '%s WHERE %s' % (statement, ' AND '.join(queries))
    return statement, params


# Querying

def forwards_relationship(src, relation):
    """
    Select destination nodes given a relation and
    a source node.

    :param src: An integer source node
    :param relation: The relation to query
    """
    statement = 'SELECT dst FROM %s WHERE src = ?'
    return statement % (relation), (src,)


def inverse_relationship(dst, relation):
    """
    Select source nodes given a destinatio node
    and a relation.

    :param dst: An integer destination node
    :param relation: The relation to query
    """
    statement = 'SELECT src FROM %s WHERE dst = ?'
    return statement % (relation), (dst,)


def get_one_relation(src, relation, dst):
    """
    Try to select one ``id`` column from a given edge.
    All arguments must be specified.

    :param src: The source node.
    :param relation: The relation.
    :param dst: A destination node.
    """
    statement = 'SELECT id FROM %s WHERE src = ? AND dst = ? LIMIT 1;'
    return statement % (relation), (src, dst)


def compound_forwards_query(relation, query):
    """
    Select destionation nodes from a given relation
    which source nodes include those in the SQL query.

    :param relation: The relation.
    :param query: An SQL query to be used as the
    subquery.
    """
    statement = 'SELECT dst FROM %s WHERE src IN (%s)'
    return statement % (relation, query)
